/**************************************************************************//**
 * @file        A9139M0.H 
 * @version     V1.00.00
 * $Revision:   0 $
 * $Date:       2020-11-13 $
 * @brief       Header file for A9139M0 RF SoC microcontroller.
 *
 * @note        V1.00.00: Modify DUALTIMER Define
 *
 * Copyright (C) 2020 AMICCOM Electronics Corp. All rights reserved.
 *****************************************************************************/

#ifndef __A9139M0_H__
#define __A9139M0_H__

#include "define.h"

/*
 * ==========================================================================
 * ---------- Interrupt Number Definition -----------------------------------
 * ==========================================================================
*/

/** @addtogroup CMSIS Device Definitions for CMSIS
  Interrupt Number Definition and Configurations for CMSIS
  @{
*/

/**
 * @details  Interrupt Number Definition. The maximum of 32 Specific Interrupts are possible.
 */

/********************************************************************
** RFLIB_Reset
** Description:		RF chip reset
** Parameters:		none
** Return value: 	none
** Note: 
*********************************************************************/
void A9139M0_Reg_Write ( Uint32 reg_addr, Uint32 write_data );

typedef enum IRQn
{
    /******  Cortex-M0 Processor Exceptions Numbers *************************************************/

    /* ToDo: use this Cortex interrupt numbers if your device is a CORTEX-M0 device                 */
    NonMaskableInt_IRQn           = -14,    /*!<  2 Cortex-M0 Non Maskable Interrupt                */
    HardFault_IRQn                = -13,    /*!<  3 Cortex-M0 Hard Fault Interrupt                  */
    SVCall_IRQn                   = -5,     /*!< 11 Cortex-M0 SV Call Interrupt                     */
    PendSV_IRQn                   = -2,     /*!< 14 Cortex-M0 Pend SV Interrupt                     */
    SysTick_IRQn                  = -1,     /*!< 15 Cortex-M0 System Tick Interrupt                 */

    /******  AMICCOM M0 Specific Interrupt Numbers **************************************************/
    Reserved0_IRQn                = 0,
    UART0_IRQn                    = 1,      /*!< UART 0 Interrupt                                   */
    SLPTIMER0_IRQn                = 2,      /*!< SLP TIMER 0 Interrupt                              */
    RADIO_IRQn                    = 3,      /*!< RADIO Interrupt                                    */
    Reserved4_IRQn                = 4,      /*!<                                                    */
    UART2_IRQn                    = 5,      /*!< UART 2 Interrupt                                   */
    GPIO0_IRQn                    = 6,      /*!< GPIO0 combined Interrupt                           */
    GPIO1_IRQn                    = 7,      /*!< GPIO1 combined Interrupt                           */
    TIMER0_IRQn                   = 8,      /*!< TIMER 0 Interrupt                                  */
    TIMER1_IRQn                   = 9,      /*!< TIMER 1 Interrupt                                  */
    DUALTIMER_IRQn                = 10,     /*!< Dual Timer Interrupt                               */
    MPU_LCD_IRQn                  = 11,     /*!< LCD DMA Driver Interrupt                           */
    Reserved12_IRQn               = 12,     /*!<                                                    */
    UART1_IRQn                    = 13,     /*!< UART 1 Interrupt                                   */
    Reserved14_IRQn               = 14,     /*!<                                                    */
    WUN_IRQn                      = 15,     /*!<                                                    */
    SPI_IRQn                      = 16,     /*!< SPI Interrupt                                      */
    I2C_IRQn                      = 17,     /*!< I2C Interrupt                                      */
    RTC_IRQn                      = 18,     /*!< RTC Interrupt                                      */
    CODEC_IRQn                    = 19,     /*!< CODEC Interrupt                                    */
    AES_CCM_IRQn                  = 20,     /*!< AES CCM Interrupt                                  */
    ADC_12BIT_IRQn                = 21,     /*!< 12-BIT ADC Interrupt                               */
    RCADC_IRQn                    = 22,     /*!< RCADC Interrupt                                    */
    USB_IRQn                      = 23,     /*!< USB Interrupt                                      */
    SLPTIMER1_IRQn                = 24,     /*!< SLP TIMER 1 Interrupt                              */
    Reserved25_IRQn               = 25,     /*!<                                                    */
    Reserved26_IRQn               = 26,     /*!<                                                    */
    Reserved27_IRQn               = 27,     /*!<                                                    */
    Reserved28_IRQn               = 28,     /*!<                                                    */
    Reserved29_IRQn               = 29,     /*!<                                                    */
    Reserved30_IRQn               = 30,     /*!<                                                    */
    SOFT_IRQn                     = 31,     /*!<                                                    */
} IRQn_Type;

/*
 * ==========================================================================
 * ----------- Processor and Core Peripheral Section ------------------------
 * ==========================================================================
 */

/* Configuration of the Cortex-M0 Processor and Core Peripherals */
#define __CM0_REV                 0x0000    /*!< Core Revision r0p0                               */
#define __NVIC_PRIO_BITS          2         /*!< Number of Bits used for Priority Levels          */
#define __Vendor_SysTickConfig    0         /*!< Set to 1 if different SysTick Config is used     */
#define __MPU_PRESENT             0         /*!< MPU present or not                               */

/*@}*/ /* end of group CMSDK_CMSIS */

#include "core_cm0.h"                       /* Cortex-M0 processor and core peripherals  */

#if defined ( __CC_ARM   )
#pragma anon_unions
#endif


typedef struct
{
    __IO uint32_t  P00     :  1;               /*!< Pn.0 */
    __IO uint32_t  P01     :  1;               /*!< Pn.1 */
    __IO uint32_t  P02     :  1;               /*!< Pn.2 */
    __IO uint32_t  P03     :  1;               /*!< Pn.3 */
    __IO uint32_t  P04     :  1;               /*!< Pn.4 */
    __IO uint32_t  P05     :  1;               /*!< Pn.5 */
    __IO uint32_t  P06     :  1;               /*!< Pn.6 */
    __IO uint32_t  P07     :  1;               /*!< Pn.7 */
    __IO uint32_t  P08     :  1;               /*!< Pn.8 */
    __IO uint32_t  P09     :  1;               /*!< Pn.9 */
    __IO uint32_t  P10     :  1;               /*!< Pn.10 */
    __IO uint32_t  P11     :  1;               /*!< Pn.11 */
    __IO uint32_t  P12     :  1;               /*!< Pn.12 */
    __IO uint32_t  P13     :  1;               /*!< Pn.13 */
    __IO uint32_t  P14     :  1;               /*!< Pn.14 */
    __IO uint32_t  P15     :  1;               /*!< Pn.15 */
    __IO uint32_t  P16     :  1;               /*!< Pn.16 */
    __IO uint32_t  P17     :  1;               /*!< Pn.17 */
    __IO uint32_t  P18     :  1;               /*!< Pn.18 */
    __IO uint32_t  P19     :  1;               /*!< Pn.19 */
    __IO uint32_t  P20     :  1;               /*!< Pn.20 */
    __IO uint32_t  P21     :  1;               /*!< Pn.21 */
    __IO uint32_t  P22     :  1;               /*!< Pn.22 */
    __IO uint32_t  P23     :  1;               /*!< Pn.23 */
    __IO uint32_t  P24     :  1;               /*!< Pn.24 */
    __IO uint32_t  P25     :  1;               /*!< Pn.25 */
    __IO uint32_t  P26     :  1;               /*!< Pn.26 */
    __IO uint32_t  P27     :  1;               /*!< Pn.27 */
    __IO uint32_t  P28     :  1;               /*!< Pn.28 */
    __IO uint32_t  P29     :  1;               /*!< Pn.29 */
    __IO uint32_t  P30     :  1;               /*!< Pn.30 */
    __IO uint32_t  P31     :  1;               /*!< Pn.31 */
} sPin32;


/*------------- Universal Asynchronous Receiver Transmitter (UART) -----------*/
/** @addtogroup UART Universal Asynchronous Receiver/Transmitter
    memory mapped structure for UART
    @{
*/
typedef struct
{
    __IO   uint32_t  DATA;          /*!< Offset: 0x000 Data Register    (R/W) */
    __IO   uint32_t  STATE;         /*!< Offset: 0x004 Status Register  (R/W) */
    __IO   uint32_t  CTRL;          /*!< Offset: 0x008 Control Register (R/W) */
    union {
        __I    uint32_t  INTSTATUS;   /*!< Offset: 0x00C Interrupt Status Register (R/ ) */
        __O    uint32_t  INTCLEAR;    /*!< Offset: 0x00C Interrupt Clear Register ( /W) */
    };
        __IO   uint32_t  BAUDDIV;       /*!< Offset: 0x010 Baudrate Divider Register (R/W) */
} UART_Type;

/* UART DATA Register Definitions */

#define UART_DATA_Pos               0                                             /*!< UART_DATA_Pos: DATA Position */
#define UART_DATA_Msk              (0xFFul << UART_DATA_Pos)                /*!< UART DATA: DATA Mask */

#define UART_STATE_RXOR_Pos         3                                             /*!< UART STATE: RXOR Position */
#define UART_STATE_RXOR_Msk         (0x1ul << UART_STATE_RXOR_Pos)          /*!< UART STATE: RXOR Mask */

#define UART_STATE_TXOR_Pos         2                                             /*!< UART STATE: TXOR Position */
#define UART_STATE_TXOR_Msk         (0x1ul << UART_STATE_TXOR_Pos)          /*!< UART STATE: TXOR Mask */

#define UART_STATE_RXBF_Pos         1                                             /*!< UART STATE: RXBF Position */
#define UART_STATE_RXBF_Msk         (0x1ul << UART_STATE_RXBF_Pos)          /*!< UART STATE: RXBF Mask */

#define UART_STATE_TXBF_Pos         0                                             /*!< UART STATE: TXBF Position */
#define UART_STATE_TXBF_Msk         (0x1ul << UART_STATE_TXBF_Pos )         /*!< UART STATE: TXBF Mask */

#define UART_CTRL_HSTM_Pos          6                                             /*!< UART CTRL: HSTM Position */
#define UART_CTRL_HSTM_Msk          (0x01ul << UART_CTRL_HSTM_Pos)          /*!< UART CTRL: HSTM Mask */

#define UART_CTRL_RXORIRQEN_Pos     5                                             /*!< UART CTRL: RXORIRQEN Position */
#define UART_CTRL_RXORIRQEN_Msk     (0x01ul << UART_CTRL_RXORIRQEN_Pos)     /*!< UART CTRL: RXORIRQEN Mask */

#define UART_CTRL_TXORIRQEN_Pos     4                                             /*!< UART CTRL: TXORIRQEN Position */
#define UART_CTRL_TXORIRQEN_Msk     (0x01ul << UART_CTRL_TXORIRQEN_Pos)     /*!< UART CTRL: TXORIRQEN Mask */

#define UART_CTRL_RXIRQEN_Pos       3                                             /*!< UART CTRL: RXIRQEN Position */
#define UART_CTRL_RXIRQEN_Msk       (0x01ul << UART_CTRL_RXIRQEN_Pos)       /*!< UART CTRL: RXIRQEN Mask */

#define UART_CTRL_TXIRQEN_Pos       2                                             /*!< UART CTRL: TXIRQEN Position */
#define UART_CTRL_TXIRQEN_Msk       (0x01ul << UART_CTRL_TXIRQEN_Pos)       /*!< UART CTRL: TXIRQEN Mask */

#define UART_CTRL_RXEN_Pos          1                                             /*!< UART CTRL: RXEN Position */
#define UART_CTRL_RXEN_Msk          (0x01ul << UART_CTRL_RXEN_Pos)          /*!< UART CTRL: RXEN Mask */

#define UART_CTRL_TXEN_Pos          0                                             /*!< UART CTRL: TXEN Position */
#define UART_CTRL_TXEN_Msk          (0x01ul << UART_CTRL_TXEN_Pos)          /*!< UART CTRL: TXEN Mask */

#define UART_INTSTATUS_RXORIRQ_Pos  3                                             /*!< UART CTRL: RXORIRQ Position */
#define UART_CTRL_RXORIRQ_Msk       (0x01ul << UART_INTSTATUS_RXORIRQ_Pos)  /*!< UART CTRL: RXORIRQ Mask */

#define UART_CTRL_TXORIRQ_Pos       2                                             /*!< UART CTRL: TXORIRQ Position */
#define UART_CTRL_TXORIRQ_Msk       (0x01ul << UART_CTRL_TXORIRQ_Pos)       /*!< UART CTRL: TXORIRQ Mask */

#define UART_CTRL_RXIRQ_Pos         1                                             /*!< UART CTRL: RXIRQ Position */
#define UART_CTRL_RXIRQ_Msk         (0x01ul << UART_CTRL_RXIRQ_Pos)         /*!< UART CTRL: RXIRQ Mask */

#define UART_CTRL_TXIRQ_Pos         0                                             /*!< UART CTRL: TXIRQ Position */
#define UART_CTRL_TXIRQ_Msk         (0x01ul << UART_CTRL_TXIRQ_Pos)         /*!< UART CTRL: TXIRQ Mask */

#define UART_BAUDDIV_Pos            0                                             /*!< UART BAUDDIV: BAUDDIV Position */
#define UART_BAUDDIV_Msk            (0xFFFFFul << UART_BAUDDIV_Pos)         /*!< UART BAUDDIV: BAUDDIV Mask */

/*@}*/ /* end of group UART */

/*----------------------------- Timer (TIMER) -------------------------------*/
/** @addtogroup TIMER
    memory mapped structure for Timer
    @{
*/
typedef struct
{
  __IO   uint32_t  CTRL;          /*!< Offset: 0x000 Control Register (R/W) */
  __IO   uint32_t  VALUE;         /*!< Offset: 0x004 Current Value Register (R/W) */
  __IO   uint32_t  RELOAD;        /*!< Offset: 0x008 Reload Value Register  (R/W) */
  union {
    __I    uint32_t  INTSTATUS;   /*!< Offset: 0x00C Interrupt Status Register (R/ ) */
    __O    uint32_t  INTCLEAR;    /*!< Offset: 0x00C Interrupt Clear Register ( /W) */
    };

} TIMER_Type;

/* TIMER CTRL Register Definitions */

#define TIMER_CTRL_IRQEN_Pos          3                                              /*!< TIMER CTRL: IRQEN Position */
#define TIMER_CTRL_IRQEN_Msk          (0x01ul << TIMER_CTRL_IRQEN_Pos)         /*!< TIMER CTRL: IRQEN Mask */

#define TIMER_CTRL_SELEXTCLK_Pos      2                                              /*!< TIMER CTRL: SELEXTCLK Position */
#define TIMER_CTRL_SELEXTCLK_Msk      (0x01ul << TIMER_CTRL_SELEXTCLK_Pos)     /*!< TIMER CTRL: SELEXTCLK Mask */

#define TIMER_CTRL_SELEXTEN_Pos       1                                              /*!< TIMER CTRL: SELEXTEN Position */
#define TIMER_CTRL_SELEXTEN_Msk       (0x01ul << TIMER_CTRL_SELEXTEN_Pos)      /*!< TIMER CTRL: SELEXTEN Mask */

#define TIMER_CTRL_EN_Pos             0                                              /*!< TIMER CTRL: EN Position */
#define TIMER_CTRL_EN_Msk             (0x01ul << TIMER_CTRL_EN_Pos)            /*!< TIMER CTRL: EN Mask */

#define TIMER_VAL_CURRENT_Pos         0                                              /*!< TIMER VALUE: CURRENT Position */
#define TIMER_VAL_CURRENT_Msk         (0xFFFFFFFFul << TIMER_VAL_CURRENT_Pos)  /*!< TIMER VALUE: CURRENT Mask */

#define TIMER_RELOAD_VAL_Pos          0                                              /*!< TIMER RELOAD: RELOAD Position */
#define TIMER_RELOAD_VAL_Msk          (0xFFFFFFFFul << TIMER_RELOAD_VAL_Pos)   /*!< TIMER RELOAD: RELOAD Mask */

#define TIMER_INTSTATUS_Pos           0                                              /*!< TIMER INTSTATUS: INTSTATUSPosition */
#define TIMER_INTSTATUS_Msk           (0x01ul << TIMER_INTSTATUS_Pos)          /*!< TIMER INTSTATUS: INTSTATUSMask */

#define TIMER_INTCLEAR_Pos            0                                              /*!< TIMER INTCLEAR: INTCLEAR Position */
#define TIMER_INTCLEAR_Msk            (0x01ul << TIMER_INTCLEAR_Pos)           /*!< TIMER INTCLEAR: INTCLEAR Mask */

/*@}*/ /* end of group TIMER */


/** @addtogroup DUALTIMER Dual Timer
    @{
*/

typedef struct
{
  __IO uint32_t LOAD;       // <h> Timer Load </h>
  __I  uint32_t VALUE;      // <h> Timer Counter Current Value <r></h>
  __IO uint32_t CTRL;       // <h> Timer Control
                                //   <o.7> TimerEn: Timer Enable
                                //   <o.6> TimerMode: Timer Mode
                                //     <0=> Freerunning-mode
                                //     <1=> Periodic mode
                                //   <o.5> IntEnable: Interrupt Enable
                                //   <o.2..3> TimerPre: Timer Prescale
                                //     <0=> / 1
                                //     <1=> / 16
                                //     <2=> / 256
                                //     <3=> Undefined!
                                //   <o.1> TimerSize: Timer Size
                                //     <0=> 16-bit counter
                                //     <1=> 32-bit counter
                                //   <o.0> OneShot: One-shoot mode
                                //     <0=> Wrapping mode
                                //     <1=> One-shot mode
                            // </h>
  __O  uint32_t INTCLR;     // <h> Timer Interrupt Clear <w></h>
  __I  uint32_t RIS;        // <h> Timer Raw Interrupt Status <r></h>
  __I  uint32_t MIS;        // <h> Timer Masked Interrupt Status <r></h>
  __IO uint32_t BGLOAD;     // <h> Background Load Register </h>
} DUALTIMER_Type;

#define DUALTIMER_LOAD_Pos             0                                            /*!< DUALTIMER LOAD: LOAD Position */
#define DUALTIMER_LOAD_Msk             (0xFFFFFFFFul << DUALTIMER_LOAD_Pos)         /*!< DUALTIMER LOAD: LOAD Mask */

#define DUALTIMER_VALUE_Pos            0                                            /*!< DUALTIMER VALUE: VALUE Position */
#define DUALTIMER_VALUE_Msk            (0xFFFFFFFFul << DUALTIMER_VALUE_Pos)        /*!< DUALTIMER VALUE: VALUE Mask */

#define DUALTIMER_CTRL_EN_Pos          7                                            /*!< DUALTIMER CTRL_EN: CTRL Enable Position */
#define DUALTIMER_CTRL_EN_Msk          (0x1ul << DUALTIMER_CTRL_EN_Pos)             /*!< DUALTIMER CTRL_EN: CTRL Enable Mask */

#define DUALTIMER_CTRL_MODE_Pos        6                                            /*!< DUALTIMER CTRL_MODE: CTRL MODE Position */
#define DUALTIMER_CTRL_MODE_Msk        (0x1ul << DUALTIMER_CTRL_MODE_Pos)           /*!< DUALTIMER CTRL_MODE: CTRL MODE Mask */

#define DUALTIMER_CTRL_INTEN_Pos       5                                            /*!< DUALTIMER CTRL_INTEN: CTRL Int Enable Position */
#define DUALTIMER_CTRL_INTEN_Msk       (0x1ul << DUALTIMER_CTRL_INTEN_Pos)          /*!< DUALTIMER CTRL_INTEN: CTRL Int Enable Mask */

#define DUALTIMER_CTRL_PRESCALE_Pos    2                                            /*!< DUALTIMER CTRL_PRESCALE: CTRL PRESCALE Position */
#define DUALTIMER_CTRL_PRESCALE_Msk    (0x3ul << DUALTIMER_CTRL_PRESCALE_Pos)       /*!< DUALTIMER CTRL_PRESCALE: CTRL PRESCALE Mask */

#define DUALTIMER_CTRL_SIZE_Pos        1                                            /*!< DUALTIMER CTRL_SIZE: CTRL SIZE Position */
#define DUALTIMER_CTRL_SIZE_Msk        (0x1ul << DUALTIMER_CTRL_SIZE_Pos)           /*!< DUALTIMER CTRL_SIZE: CTRL SIZE Mask */

#define DUALTIMER_CTRL_ONESHOOT_Pos    0                                            /*!< DUALTIMER CTRL_ONESHOOT: CTRL ONESHOOT Position */
#define DUALTIMER_CTRL_ONESHOOT_Msk    (0x1ul << DUALTIMER_CTRL_ONESHOOT_Pos)       /*!< DUALTIMER CTRL_ONESHOOT: CTRL ONESHOOT Mask */

#define DUALTIMER_INTCLR_Pos           0                                            /*!< DUALTIMER INTCLR: INT Clear Position */
#define DUALTIMER_INTCLR_Msk           (0x1ul << DUALTIMER_INTCLR_Pos)              /*!< DUALTIMER INTCLR: INT Clear  Mask */

#define DUALTIMER_RAWINTSTAT_Pos       0                                            /*!< DUALTIMER RAWINTSTAT: Raw Int Status Position */
#define DUALTIMER_RAWINTSTAT_Msk       (0x1ul << DUALTIMER_RAWINTSTAT_Pos)          /*!< DUALTIMER RAWINTSTAT: Raw Int Status Mask */

#define DUALTIMER_MASKINTSTAT_Pos      0                                            /*!< DUALTIMER MASKINTSTAT: Mask Int Status Position */
#define DUALTIMER_MASKINTSTAT_Msk      (0x1ul << DUALTIMER_MASKINTSTAT_Pos)         /*!< DUALTIMER MASKINTSTAT: Mask Int Status Mask */

#define DUALTIMER_BGLOAD_Pos           0                                            /*!< DUALTIMER BGLOAD: Background Load Position */
#define DUALTIMER_BGLOAD_Msk           (0xFFFFFFFFul << DUALTIMER_BGLOAD_Pos)       /*!< DUALTIMER BGLOAD: Background Load Mask */

/*@}*/ /* end of group DUALTIMER */


/*-------------------- General Purpose Input Output (GPIO) -------------------*/

/** @addtogroup GPIO GPIO
    @{
*/

typedef struct
{
    union {
        __IO    uint32_t    DATA;           /*!< Offset: 0x000  DATA Register (R/W)                             */
        __IO    sPin32      DATA_PIN;       /*!< Offset: 0x000  Data Register access by Pin                     */
    };
    union {
        __IO    uint32_t    DATAOUT;        /*!< Offset: 0x004  Data Output Latch Register (R/W) */
        __IO    sPin32      DATAOUT_PIN;    /*!< Offset: 0x004  GPIO Output Data Register access by Pin         */
    };
    union {
        __IO    uint32_t    PUN;            /*!< Offset: 0x008  GPIO Pull-Up Resistor NOT Register              */
        __IO    sPin32      PUN_PIN;        /*!< Offset: 0x008  GPIO Pull-Up Resistor NOT Register access by pin*/
    };
    union {
        __IO    uint32_t    WUN;            /*!< Offset: 0x00C  GPIO Wakeup NOT Register                        */
        __IO    sPin32      WUN_PIN;        /*!< Offset: 0x00C  GPIO Wakeup NOT Register access by pin          */
    };
    __IO    uint32_t    OUTENABLESET;       /*!< Offset: 0x010  Output Enable Set Register  (R/W)               */
    __IO    uint32_t    OUTENABLECLR;       /*!< Offset: 0x014  Output Enable Clear Register  (R/W)             */
    __IO    uint32_t    ALTFUNCSET;         /*!< Offset: 0x018  Alternate Function Set Register  (R/W)          */
    __IO    uint32_t    ALTFUNCCLR;         /*!< Offset: 0x01C  Alternate Function Clear Register  (R/W)        */
    __IO    uint32_t    INTENSET;           /*!< Offset: 0x020  Interrupt Enable Set Register  (R/W)            */
    __IO    uint32_t    INTENCLR;           /*!< Offset: 0x024  Interrupt Enable Clear Register  (R/W)          */
    __IO    uint32_t    INTTYPESET;         /*!< Offset: 0x028  Interrupt Type Set Register  (R/W)              */
    __IO    uint32_t    INTTYPECLR;         /*!< Offset: 0x02C  Interrupt Type Clear Register  (R/W)            */
    __IO    uint32_t    INTPOLSET;          /*!< Offset: 0x030  Interrupt Polarity Set Register  (R/W)          */
    __IO    uint32_t    INTPOLCLR;          /*!< Offset: 0x034  Interrupt Polarity Clear Register  (R/W)        */
    union {
        __I     uint32_t    INTSTATUS;      /*!< Offset: 0x038  Interrupt Status Register (R/ )                 */
        __O     uint32_t    INTCLEAR;       /*!< Offset: 0x038  Interrupt Clear Register ( /W)                  */
    };
            uint32_t    RESERVED1[241];
    __IO    uint32_t    LB_MASKED[256];     /*!< Offset: 0x400 - 0x7FC Lower byte Masked Access Register (R/W)  */
    __IO    uint32_t    UB_MASKED[256];     /*!< Offset: 0x800 - 0xBFC Upper byte Masked Access Register (R/W)  */
} GPIO_Type;

#define GPIO_DATA_Pos           0                                          /*!< GPIO DATA: DATA Position */
#define GPIO_DATA_Msk           (0xFFFFFFFFul << GPIO_DATA_Pos)          /*!< GPIO DATA: DATA Mask */

#define GPIO_DATAOUT_Pos        0                                          /*!< GPIO DATAOUT: DATAOUT Position */
#define GPIO_DATAOUT_Msk        (0xFFFFFFFFul << GPIO_DATAOUT_Pos)       /*!< GPIO DATAOUT: DATAOUT Mask */

#define GPIO_PUN_Pos            0                                          /*!< GPIO DATAOUT: DATAOUT Position */
#define GPIO_PUN_Msk            (0xFFFFFFFFul << GPIO_PUN_Pos)       /*!< GPIO DATAOUT: DATAOUT Mask */

#define GPIO_WUN_Pos            0                                          /*!< GPIO DATAOUT: DATAOUT Position */
#define GPIO_WUN_Msk            (0xFFFFFFFFul << GPIO_WUN_Pos)       /*!< GPIO DATAOUT: DATAOUT Mask */

#define GPIO_OUTENSET_Pos       0                                          /*!< GPIO OUTEN: OUTEN Position */
#define GPIO_OUTENSET_Msk       (0xFFFFFFFFul << GPIO_OUTEN_Pos)         /*!< GPIO OUTEN: OUTEN Mask */

#define GPIO_OUTENCLR_Pos       0                                          /*!< GPIO OUTEN: OUTEN Position */
#define GPIO_OUTENCLR_Msk       (0xFFFFFFFFul << GPIO_OUTEN_Pos)         /*!< GPIO OUTEN: OUTEN Mask */

#define GPIO_ALTFUNCSET_Pos     0                                          /*!< GPIO ALTFUNC: ALTFUNC Position */
#define GPIO_ALTFUNCSET_Msk     (0xFFFFFFFFul << GPIO_ALTFUNC_Pos)       /*!< GPIO ALTFUNC: ALTFUNC Mask */

#define GPIO_ALTFUNCCLR_Pos     0                                          /*!< GPIO ALTFUNC: ALTFUNC Position */
#define GPIO_ALTFUNCCLR_Msk     (0xFFFFFFFFul << GPIO_ALTFUNC_Pos)       /*!< GPIO ALTFUNC: ALTFUNC Mask */

#define GPIO_INTENSET_Pos       0                                          /*!< GPIO INTEN: INTEN Position */
#define GPIO_INTENSET_Msk       (0xFFFFFFFFul << GPIO_INTEN_Pos)         /*!< GPIO INTEN: INTEN Mask */

#define GPIO_INTENCLR_Pos       0                                          /*!< GPIO INTEN: INTEN Position */
#define GPIO_INTENCLR_Msk       (0xFFFFFFFFul << GPIO_INTEN_Pos)         /*!< GPIO INTEN: INTEN Mask */

#define GPIO_INTTYPESET_Pos     0                                          /*!< GPIO INTTYPE: INTTYPE Position */
#define GPIO_INTTYPESET_Msk     (0xFFFFFFFFul << GPIO_INTTYPE_Pos)       /*!< GPIO INTTYPE: INTTYPE Mask */

#define GPIO_INTTYPECLR_Pos     0                                          /*!< GPIO INTTYPE: INTTYPE Position */
#define GPIO_INTTYPECLR_Msk     (0xFFFFFFFFul << GPIO_INTTYPE_Pos)       /*!< GPIO INTTYPE: INTTYPE Mask */

#define GPIO_INTPOLSET_Pos      0                                          /*!< GPIO INTPOL: INTPOL Position */
#define GPIO_INTPOLSET_Msk      (0xFFFFFFFFul << GPIO_INTPOL_Pos)        /*!< GPIO INTPOL: INTPOL Mask */

#define GPIO_INTPOLCLR_Pos      0                                          /*!< GPIO INTPOL: INTPOL Position */
#define GPIO_INTPOLCLR_Msk      (0xFFFFFFFFul << GPIO_INTPOL_Pos)        /*!< GPIO INTPOL: INTPOL Mask */

#define GPIO_INTSTATUS_Pos      0                                          /*!< GPIO INTSTATUS: INTSTATUS Position */
#define GPIO_INTSTATUS_Msk      (0xFFFFFFFFul << GPIO_INTSTATUS_Pos)       /*!< GPIO INTSTATUS: INTSTATUS Mask */

#define GPIO_INTCLEAR_Pos       0                                          /*!< GPIO INTCLEAR: INTCLEAR Position */
#define GPIO_INTCLEAR_Msk       (0xFFFFFFFFul << GPIO_INTCLEAR_Pos)        /*!< GPIO INTCLEAR: INTCLEAR Mask */



/*@}*/ /* end of group GPIO */

/*------------- System Control (SYSCON) --------------------------------------*/
/** @addtogroup SYSCON System Control
    @{
*/
typedef struct
{
    __IO   uint32_t  REMAP;          /*!< Offset: 0x000 Remap Control Register (R/W) */
    __IO   uint32_t  PMUCTRL;        /*!< Offset: 0x004 PMU Control Register (R/W) */
    __IO   uint32_t  RESETOP;        /*!< Offset: 0x008 Reset Option Register  (R/W) */
    __IO   uint32_t  EMICTRL;        /*!< Offset: 0x00C EMI Control Register  (R/W) */
    __IO   uint32_t  RSTINFO;        /*!< Offset: 0x010 Reset Information Register (R/W) */
} SYSCON_Type;

#define SYSCON_REMAP_Pos                 0
#define SYSCON_REMAP_Msk                 (0x01ul << SYSCON_REMAP_Pos)               /*!< SYSCON MEME_CTRL: REMAP Mask */

#define SYSCON_PMUCTRL_EN_Pos            0
#define SYSCON_PMUCTRL_EN_Msk            (0x01ul << SYSCON_PMUCTRL_EN_Pos)          /*!< SYSCON PMUCTRL: PMUCTRL ENABLE Mask */

#define SYSCON_LOCKUPRST_RESETOP_Pos     0
#define SYSCON_LOCKUPRST_RESETOP_Msk     (0x01ul << SYSCON_LOCKUPRST_RESETOP_Pos)   /*!< SYSCON SYS_CTRL: LOCKUP RESET ENABLE Mask */

#define SYSCON_EMICTRL_SIZE_Pos          24
#define SYSCON_EMICTRL_SIZE_Msk          (0x00001ul << SYSCON_EMICTRL_SIZE_Pos)     /*!< SYSCON EMICTRL: SIZE Mask */

#define SYSCON_EMICTRL_TACYC_Pos         16
#define SYSCON_EMICTRL_TACYC_Msk         (0x00007ul << SYSCON_EMICTRL_TACYC_Pos)    /*!< SYSCON EMICTRL: TURNAROUNDCYCLE Mask */

#define SYSCON_EMICTRL_WCYC_Pos          8
#define SYSCON_EMICTRL_WCYC_Msk          (0x00003ul << SYSCON_EMICTRL_WCYC_Pos)     /*!< SYSCON EMICTRL: WRITECYCLE Mask */

#define SYSCON_EMICTRL_RCYC_Pos          0
#define SYSCON_EMICTRL_RCYC_Msk          (0x00007ul << SYSCON_EMICTRL_RCYC_Pos)     /*!< SYSCON EMICTRL: READCYCLE Mask */

#define SYSCON_RSTINFO_SYSRESETREQ_Pos   0
#define SYSCON_RSTINFO_SYSRESETREQ_Msk   (0x00001ul << SYSCON_RSTINFO_SYSRESETREQ_Pos) /*!< SYSCON RSTINFO: SYSRESETREQ Mask */

#define SYSCON_RSTINFO_WDOGRESETREQ_Pos  1
#define SYSCON_RSTINFO_WDOGRESETREQ_Msk  (0x00001ul << SYSCON_RSTINFO_WDOGRESETREQ_Pos) /*!< SYSCON RSTINFO: WDOGRESETREQ Mask */

#define SYSCON_RSTINFO_LOCKUPRESET_Pos   2
#define SYSCON_RSTINFO_LOCKUPRESET_Msk   (0x00001ul << SYSCON_RSTINFO_LOCKUPRESET_Pos) /*!< SYSCON RSTINFO: LOCKUPRESET Mask */

/*@}*/ /* end of group SYSCON */

/*------------------- Watchdog ----------------------------------------------*/
/** @addtogroup WATCHDOG Watchdog
    @{
*/
typedef struct
{
    __IO    uint32_t  LOAD;         // <h> Watchdog Load Register </h>
    __I     uint32_t  VALUE;        // <h> Watchdog Value Register </h>
    __IO    uint32_t  CTRL;         // <h> Watchdog Control Register
                                    //   <o.1>    RESEN: Reset enable
                                    //   <o.0>    INTEN: Interrupt enable
                                    // </h>
    __O     uint32_t  INTCLR;       // <h> Watchdog Clear Interrupt Register </h>
    __I     uint32_t  RAWINTSTAT;   // <h> Watchdog Raw Interrupt Status Register </h>
    __I     uint32_t  MASKINTSTAT;  // <h> Watchdog Interrupt Status Register </h>
        uint32_t  RESERVED0[762];
    __IO    uint32_t  LOCK;         // <h> Watchdog Lock Register </h>
        uint32_t  RESERVED1[191];
    __IO    uint32_t  ITCR;         // <h> Watchdog Integration Test Control Register </h>
    __O     uint32_t  ITOP;         // <h> Watchdog Integration Test Output Set Register </h>
} WATCHDOG_Type;

#define WATCHDOG_LOAD_Pos               0                                              /*!< WATCHDOG LOAD: LOAD Position */
#define WATCHDOG_LOAD_Msk              (0xFFFFFFFFul << WATCHDOG_LOAD_Pos)       /*!< WATCHDOG LOAD: LOAD Mask */

#define WATCHDOG_VALUE_Pos              0                                              /*!< WATCHDOG VALUE: VALUE Position */
#define WATCHDOG_VALUE_Msk             (0xFFFFFFFFul << WATCHDOG_VALUE_Pos)      /*!< WATCHDOG VALUE: VALUE Mask */

#define WATCHDOG_CTRL_RESEN_Pos         1                                              /*!< WATCHDOG CTRL_RESEN: Enable Reset Output Position */
#define WATCHDOG_CTRL_RESEN_Msk        (0x1ul << WATCHDOG_CTRL_RESEN_Pos)        /*!< WATCHDOG CTRL_RESEN: Enable Reset Output Mask */

#define WATCHDOG_CTRL_INTEN_Pos         0                                              /*!< WATCHDOG CTRL_INTEN: Int Enable Position */
#define WATCHDOG_CTRL_INTEN_Msk        (0x1ul << WATCHDOG_CTRL_INTEN_Pos)        /*!< WATCHDOG CTRL_INTEN: Int Enable Mask */

#define WATCHDOG_INTCLR_Pos             0                                              /*!< WATCHDOG INTCLR: Int Clear Position */
#define WATCHDOG_INTCLR_Msk            (0x1ul << WATCHDOG_INTCLR_Pos)            /*!< WATCHDOG INTCLR: Int Clear Mask */

#define WATCHDOG_RAWINTSTAT_Pos         0                                              /*!< WATCHDOG RAWINTSTAT: Raw Int Status Position */
#define WATCHDOG_RAWINTSTAT_Msk        (0x1ul << WATCHDOG_RAWINTSTAT_Pos)        /*!< WATCHDOG RAWINTSTAT: Raw Int Status Mask */

#define WATCHDOG_MASKINTSTAT_Pos        0                                              /*!< WATCHDOG MASKINTSTAT: Mask Int Status Position */
#define WATCHDOG_MASKINTSTAT_Msk       (0x1ul << WATCHDOG_MASKINTSTAT_Pos)       /*!< WATCHDOG MASKINTSTAT: Mask Int Status Mask */

#define WATCHDOG_LOCK_Pos               0                                              /*!< WATCHDOG LOCK: LOCK Position */
#define WATCHDOG_LOCK_Msk              (0x1ul << WATCHDOG_LOCK_Pos)              /*!< WATCHDOG LOCK: LOCK Mask */

#define WATCHDOG_INTEGTESTEN_Pos        0                                              /*!< WATCHDOG INTEGTESTEN: Integration Test Enable Position */
#define WATCHDOG_INTEGTESTEN_Msk       (0x1ul << WATCHDOG_INTEGTESTEN_Pos)       /*!< WATCHDOG INTEGTESTEN: Integration Test Enable Mask */

#define WATCHDOG_INTEGTESTOUTSET_Pos    1                                              /*!< WATCHDOG INTEGTESTOUTSET: Integration Test Output Set Position */
#define WATCHDOG_INTEGTESTOUTSET_Msk   (0x1ul << WATCHDOG_INTEGTESTOUTSET_Pos)   /*!< WATCHDOG INTEGTESTOUTSET: Integration Test Output Set Mask */

/*@}*/ /* end of group  WATCHDOG */


/*------------------- PWM ----------------------------------------------*/
/** @addtogroup PWM Pulse Width Modulation Controller
    @{
*/
typedef struct
{
    __IO    uint32_t    CTRL;       // PWM control register
                                    //      BITS[0:2] - Prescal by power(2, BITS[0:2])
                                    //      BITS[3]   - clock source select. 0=system clock, 1=RTC or IRC
                                    //      BITS[7]   - Enable PWM
    __IO    uint32_t    HIGH_LOW;   //  PWM set high/low register
                                    //      BITS[0:7] - Set LOW width by 256-LOW
                                    //      BITS[8:16]- Set HIGH wdith by 256-HIGH
                                    //      The HIGH must large then LOW.
} PWM_Type;

#define PWM_CTRL_PRESCAL_Pos        0
#define PWM_CTRL_PRESCAL_Msk        (0x7ul << PWM_CTRL_PRESCAL_Pos)

#define PWM_CTRL_CKS_Pos            3
#define PWM_CTRL_CKS_Msk            (0x1ul << PWM_CTRL_CKS_Pos)

#define PWM_CTRL_EN_Pos             7
#define PWM_CTRL_EN_Msk             (0x1ul << PWM_CTRL_EN_Pos)

#define PWM_HIGH_LOW_LOW_Pos        0
#define PWM_HIGH_LOW_LOW_Msk        (0xFFul << PWM_HIGH_LOW_LOW_Pos)

#define PWM_HIGH_LOW_HIGH_Pos       8
#define PWM_HIGH_LOW_HIGH_Msk       (0xFFul << PWM_HIGH_LOW_HIGH_Pos)

/*@}*/ /* end of group  PWM */

/*------------------- I2CM ----------------------------------------------*/
/** @addtogroup I2CM I2C Master
    @{
*/
typedef struct
{
    __IO    uint32_t  SLAVE_ADDR;       // Offset: 0x000    I2C Slave Address register
                                        //  [0]     - R/S(receive / not send). 0=SEND, 1=RECEIVE
                                        //  [7:1]   - 7 address bits (A6-A0)
    union {
        __O     uint32_t  CTRL;         // Offset: 0x004    I2C Master Control register
                                        //  [0]     - RUN
                                        //  [1]     - START
                                        //  [2]     - STOP
                                        //  [3]     - ACK
                                        //  [4]     - HS
                                        //  [5]     -
                                        //  [6]     - SLRST
                                        //  [7]     - RSTB
        __I     uint32_t  STATUS;       // Offset: 0x004    I2C Master Status register
                                        //  [0]     - BUSY      -
                                        //  [1]     - ERROR
                                        //  [2]     - ADDR_ACK
                                        //  [3]     - DATA_ACK
                                        //  [4]     - ARB_LOST
                                        //  [5]     - IDLE
                                        //  [6]     - BUS_BUSY
                                        //  [7]     - reserved
    };
    __IO    uint32_t  BUF;              // Offset: 0x008    I2C Master Buffer data
    __IO    uint32_t  TP;               // Offset: 0x00C    I2C Master Timer Period Register
    __IO    uint32_t  SCL_LP;           // Offset: 0x010    I2C Master SCL Low Period.
    __IO    uint32_t  SCL_HP;           // Offset: 0x014    I2C Master SCL High Period.
    __IO    uint32_t  DATA_SETUP_TIME;  // Offset: 0x018    I2C Master SDA setup time
    __IO    uint32_t  INT;              // Offset: 0x01C    I2C Master Interrupt Control
                                        //  [0]     - ENABLE    Write 1 to Enable INT. Write 0 to disable INT.
                                        //  [1]     - FLAG      Write 1 to Clean
    __O     uint32_t  IO_SEL;           // Offset: 0x020
                                        //  [0]     - SEL
} I2CM_Type;

#define I2CM_SLAVE_ADDR_RS_Pos      0
#define I2CM_SLAVE_ADDR_RS_Msk      (0x1ul<<I2CM_SLAVE_ADDR_RS_Pos)

#define I2CM_SLAVE_ADDR_ADDR_Pos    1
#define I2CM_SLAVE_ADDR_ADDR_Msk    (0x7Ful<<I2CM_SLAVE_ADDR_Pos)

#define I2CM_CTRL_RUN_Pos           0
#define I2CM_CTRL_RUN_Msk           (0x1ul<<I2CM_CTRL_RUN_Pos)

#define I2CM_CTRL_START_Pos         1
#define I2CM_CTRL_START_Msk         (0x1ul<<I2CM_CTRL_START_Pos)

#define I2CM_CTRL_STOP_Pos          2
#define I2CM_CTRL_STOP_Msk          (0x1ul<<I2CM_CTRL_STOP_Pos)

#define I2CM_CTRL_ACK_Pos           3
#define I2CM_CTRL_ACK_Msk           (0x1ul<<I2CM_CTRL_ACK_Pos)

#define I2CM_CTRL_HS_Pos            4
#define I2CM_CTRL_HS_Msk            (0x1ul<<I2CM_CTRL_HS_Pos)

#define I2CM_CTRL_ADDR_Pos          5
#define I2CM_CTRL_ADDR_Msk          (0x1ul<<I2CM_CTRL_ADDR_Pos)

#define I2CM_CTRL_SLRST_Pos         6
#define I2CM_CTRL_SLRST_Msk         (0x1ul<<I2CM_CTRL_SLRST_Pos)

#define I2CM_CTRL_RSTB_Pos          7
#define I2CM_CTRL_RSTB_Msk          (0x1ul<<I2CM_CTRL_RSTB_Pos)

#define I2CM_STATUS_BUSY_Pos        0
#define I2CM_STATUS_BUSY_Msk        (0x1ul<<I2CM_STATUS_BUSY_Pos)

#define I2CM_STATUS_ERROR_Pos       1
#define I2CM_STATUS_ERROR_Msk       (0x1ul<<I2CM_STATUS_ERROR_Pos)

#define I2CM_STATUS_ADDR_ACK_Pos    2
#define I2CM_STATUS_ADDR_ACK_Msk    (0x1ul<<I2CM_STATUS_ADDR_ACK_Pos)

#define I2CM_STATUS_DATA_ACK_Pos    3
#define I2CM_STATUS_DATA_ACK_Msk    (0x1ul<<I2CM_STATUS_DATA_ACK_Pos)

#define I2CM_STATUS_ARB_LOST_Pos    4
#define I2CM_STATUS_ARB_LOST_Msk    (0x1ul<<I2CM_STATUS_ARB_LOST_Pos)

#define I2CM_STATUS_IDLE_Pos        5
#define I2CM_STATUS_IDLE_Msk        (0x1ul<<I2CM_STATUS_IDLE_Pos)

#define I2CM_STATUS_BUS_BUSY_Pos    6
#define I2CM_STATUS_BUS_BUSY_Msk    (0x1ul<<I2CM_STATUS_BUS_BUSY_Pos)

#define I2CM_BUF_Pos                0
#define I2CM_BUF_Msk                (0xFFul<<I2CM_BUF_Pos)

#define I2CM_TP_Pos                 0
#define I2CM_TP_Msk                 (0x7Ful<<I2CM_TP_Pos)

#define I2CM_SCL_LOW_Pos            0
#define I2CM_SCL_LOW_Msk            (0xFul<<I2CM_SCL_LOW_Pos)

#define I2CM_SCL_HIGH_Pos           0
#define I2CM_SCL_HIGH_Msk           (0xFul<<I2CM_SCL_HIGH_Pos)

#define I2CM_DATA_SETUP_TIME_Pos    0
#define I2CM_DATA_SETUP_TIME_Msk    (0xFul<<I2CM_DATA_SETUP_TIME_Pos)

#define I2CM_INT_ENABLE_Pos         0
#define I2CM_INT_ENABLE_Msk         (0x1ul<<I2CM_INT_ENABLE_Pos)

#define I2CM_INT_FLAG_Pos           1
#define I2CM_INT_FLAG_Msk           (0x1ul<<I2CM_INT_FLAG_Pos)

#define I2CM_IO_SEL_Pos             0
#define I2CM_IO_SEL_Msk             (0x1ul<<I2CM_IO_SEL_Pos)

#define I2CM_CKEN_Pos               7 
#define I2CM_CKEN_Msk               (0x1ul<<I2CM_CKEN_Pos)

/*@}*/ /* end of group  I2CM */


/*------------------- I2CS ----------------------------------------------*/
/** @addtogroup I2CS I2C Slave
    @{
*/
typedef struct
{
    __IO    uint32_t  ADDR;             // Offset: 0x000    I2C Slave Address
    union {
        __O     uint32_t  CTRL;         // Offset: 0x004    I2C Slave Control
                                        //  [2]     - SENDFINCLR    Write 1 to Clean SENDFIN
                                        //  [3]     - RECFINCLR     Write 1 to Clean RECFIN
                                        //  [6]     - DA            Device Active
                                        //  [7]     - RSTB          I2CS reset bit
        __I     uint32_t  STATUS;       // Offset: 0x004    I2C Slave Status
                                        //  [0]     - RREQ          Read BUF to Clean
                                        //  [1]     - TREQ          Write BUF to Clean
                                        //  [2]     - SENDFIN       Send Finish
                                        //  [3]     - RECFIN        Receive Finish
                                        //  [4]     - BUSACTIVE     Bus actived
                                        //  [6]     - DA            Device Actived
    };
    __IO    uint32_t  BUF;              // Offset: 0x008    I2C Slave Data Buffer
    __IO    uint32_t  ADDR10BITS;       // Offset: 0x00C    I2C Slave Address 10bits extern
    __IO    uint32_t  INT;              // Offset: 0x010    I2C Slave Interrupt

} I2CS_Type;

#define I2CS_ADDR_Pos               0
#define I2CS_ADDR_Msk               (0x7Ful<<I2CS_ADDR_Pos)

#define I2CS_CTRL_SENDFINCLR_Pos    2
#define I2CS_CTRL_SENDFINCLR_Msk    (0x1ul<<I2CS_CTRL_SENDFINCLR_Pos)

#define I2CS_CTRL_RECFINCLR_Pos     3
#define I2CS_CTRL_RECFINCLR_Msk     (0x1ul<<I2CS_CTRL_RECFINCLR_Pos)

#define I2CS_CTRL_DA_Pos            6
#define I2CS_CTRL_DA_Msk            (0x1ul<<I2CS_CTRL_DA_Pos)

#define I2CS_CTRL_RSTB_Pos          7
#define I2CS_CTRL_RSTB_Msk          (0x1ul<<I2CS_CTRL_RSTB_Pos)

#define I2CS_STATUS_RREQ_Pos        0
#define I2CS_STATUS_RREQ_Msk        (0x1ul<<I2CS_STATUS_RREQ_Pos)

#define I2CS_STATUS_TREQ_Pos        1
#define I2CS_STATUS_TREQ_Msk        (0x1ul<<I2CS_STATUS_TREQ_Pos)

#define I2CS_STATUS_SENDFIN_Pos     2
#define I2CS_STATUS_SENDFIN_Msk     (0x1ul<<I2CS_STATUS_SENDFIN_Pos)

#define I2CS_STATUS_RECFIN_Pos      3
#define I2CS_STATUS_RECFIN_Msk      (0x1ul<<I2CS_STATUS_RECFIN_Pos)

#define I2CS_STATUS_BUSACTIVE_Pos   4
#define I2CS_STATUS_BUSACTIVE_Msk   (0x1ul<<I2CS_STATUS_BUSACTIVE_Pos)

#define I2CS_STATUS_DA_Pos          6
#define I2CS_STATUS_DA_Msk          (0x1ul<<I2CS_STATUS_DA_Pos)

#define I2CS_BUF_Pos                0
#define I2CS_BUF_Msk                (0xFFul<<I2CS_BUF_Pos)

#define I2CS_ADDR10BITS_A9A8A7_Pos  0
#define I2CS_ADDR10BITS_A9A8A7_Msk  (0x7ul<<I2CS_ADDR10BITS_A9A8A7_Pos)

#define I2CS_ADDR10BITS_EN_Pos      3
#define I2CS_ADDR10BITS_EN_Msk      (0x1ul<<I2CS_ADDR10BITS_EN_Pos)

#define I2CS_INT_ENABLE_Pos         0
#define I2CS_INT_ENABLE_Msk         (0x1ul<<I2CS_INT_ENABLE_Pos)

#define I2CS_INT_FLAG_Pos           1
#define I2CS_INT_FLAG_Msk           (0x1ul<<I2CS_INT_FLAG_Pos)

#define I2CS_CKEN_Pos               7 
#define I2CS_CKEN_Msk               (0x1ul<<I2CS_CKEN_Pos)

/*@}*/ /* end of group  I2CS */

/*-------------------- FLASH -------------------*/
/** @addtogroup FLASH
    memory mapped structure for FLASH
    @{
*/
typedef struct
{
    __IO    uint32_t  reserved1;                // Offset: 0x000
    __IO    uint32_t  CTRL;                     // Offset: 0x004            Flash control register
    __IO    uint32_t  LENGTH;                   // Offset: 0x008            The length by 32bits for flash action
    __IO    uint32_t  ADDR;                     // Offset: 0x00C            The address for flash action
    __IO    uint32_t  PWD0;                     // Offset: 0x010            Flash password0 register
    __IO    uint32_t  PWD1;                     // Offset: 0x014            Flash password1 register
    __IO    uint32_t  KEY0;                     // Offset: 0x018            Flash key0 register
    __IO    uint32_t  KEY1;                     // Offset: 0x01C            Flash key1 register
            uint8_t   reserved2[0x100-0x01C-4];
    union {
        __IO    uint32_t  DATA32[32];           // Offset: 0x100~0x17F      Flash Write/Program Data buffer by 32bits
        __IO    uint8_t   DATA8[128];           // Offset: 0x100~0x17C      Flash Write/Program Data buffer by BYTE
    };
            uint8_t   reserved3[0x200-0x180];
    __IO    uint32_t  LOCK[2];                  // Offset: 0x200            Flash Lock register		    
    union {
				__IO    uint32_t  WAIT_STATE;           // Offset: 0x208            Flash Wait State register
		    struct
        {
            __IO  uint32_t ws_cfg       :2;
            __IO  uint32_t ENPLL        :1;
            __IO  uint32_t IS_PLL_Mode  :1;
            __IO  uint32_t Boot_Load_Done	          :1;
            __IO  uint32_t MCUSWS       :1;
            __IO  uint32_t DBBS         :1;
            __IO  uint32_t DBRS         :1;
            __IO  uint32_t IS_CACHE_WAY2            :3;
						__IO  uint32_t RSEL         :3;
                  uint32_t WAIT_STATE_BIT_RESERVED0 :22;            
        }WAIT_STATE_Item;
    };     	
} Flash_Type;

/* FLASH Register Definitions */

#define FLASH_CTRL_WRITE_Pos    0
#define FLASH_CTRL_WRITE_Msk    (0x1ul<<FLASH_CTRL_WRITE_Pos)
#define FLASH_CTRL_PROG_Pos     1
#define FLASH_CTRL_PROG_Msk     (0x1ul<<FLASH_CTRL_PROG_Pos)
#define FLASH_CTRL_PERASE_Pos   2
#define FLASH_CTRL_PERASE_Msk   (0x1ul<<FLASH_CTRL_PERASE_Pos)
#define FLASH_CTRL_SERASE_Pos   3
#define FLASH_CTRL_SERASE_Msk   (0x1ul<<FLASH_CTRL_SERASE_Pos)
#define FLASH_CTRL_MERASE_Pos   4
#define FLASH_CTRL_MERASE_Msk   (0x1ul<<FLASH_CTRL_MERASE_Pos)
#define FLASH_CTRL_F1M_Pos      5
#define FLASH_CTRL_F1M_Msk      (0x1ul<<FLASH_CTRL_F1M_Pos)
#define FLASH_CTRL_F2M_Pos      6
#define FLASH_CTRL_F2M_Msk      (0x1ul<<FLASH_CTRL_F2M_Pos)
#define FLASH_CTRL_CE_Pos       7
#define FLASH_CTRL_CE_Msk       (0x1ul<<FLASH_CTRL_CE_Pos)

#define FLASH_LENGTH_PWE_Pos    7
#define FLASH_LENGTH_PWE_Msk    (0x1ul<<FLASH_LENGTH_PWE_Pos)

/*@}*/ /* end of group FLASH */


/*-------------------- Radio Frequency -------------------*/

/** @addtogroup RADIO
    @{
*/
#define RF_FIFO_SIZE   256

typedef struct
{
    

    __O     uint32_t  RESET;              /*!< Offset: 0x000 RF_RESET Register  (W) */

    __IO    uint32_t  MODECTRL1;             /*!< Offset: 0x004 Strobe Command Register  (R/W) */
    __IO    uint32_t  MODECTRL2;           /*!< Offset: 0x008 Mode Control Register (R/W) */
            uint32_t  RF_RESERVED0[1];    /*!< Offset: 0x00C Reserved */
    
    union
    {
        	__I     uint32_t  STATUS;                   /*!< Offset: 0x010 STATUS Register (R)  */
	        struct
	        {
	            __I uint32_t TRER				:1;
	            __I uint32_t TRSR				:1;
	            __I uint32_t PLLER				:1;
	            __I uint32_t XER				:1;
	            __I uint32_t CER				:1;
	            __I uint32_t CRCF				:1;
	            __I uint32_t FECF				:1;
	                uint32_t STATUS_RESERVED0	:1;
	            __I uint32_t ARCWTR				:1;
	            __I uint32_t ENC_FLAG			:1;
	            __I uint32_t FPF				:1;
	            __I uint32_t FSYNC				:1;
	            __I uint32_t PWR				:1;
	            __I uint32_t P_IRQ2O			:1;
	            __I uint32_t P_IRQ1O			:1;
	            __I uint32_t P_CKO				:1;
	                uint32_t STATUS_RESERVED1	:16;
	                    
	        }STATUS_ReadItem;
        
        
    };
    
    union
    {
           	__O uint32_t RFINT_WRITE;                       /*!< Offset: 0x014 INTIE Register (W)  */   
            __I uint32_t RFINT_READ;                        /*!< Offset: 0x014 INTIE Register (R)  */
            struct
            {
                __I  uint32_t WTR_INT_STATE     :1;
                __I  uint32_t WAKEBBINT_STATE   :1;
                __I  uint32_t FPFIE_STATE       :1;
                __I  uint32_t FSYNCIE_STATE     :1;
								__I  uint32_t VPOAKIE_STATE     :1;
                     uint32_t RFINT_R_RESERVED0 :1;
                __I  uint32_t RADIO_INT_CLEAR   :1;
                __I  uint32_t EXT_INT           :1;  
                     uint32_t RFINT_R_RESERVED1 :24;   

            }RFINT_READ_Item;
                
    };
        
    union
    {
        __IO   uint32_t  SYNC_ID;         /*!< Offset: 0x018 RF ID Register (R/W)  */
        __IO   uint8_t   SYNCB_ID[4];
    };
    union
    {
        __IO   uint32_t  SYNC_ID2;         /*!< Offset: 0x01C RF ID2 Register (R/W)  */
        __IO   uint8_t   SYNCB_ID2[4];
    };

    
    union
    {
        __IO    uint32_t  FIFOCTRL;           /*!< Offset: 0x020 FIFO Control Register (R/W)  */
        	struct
            {
                __IO  uint32_t FEP                      :14;
                      uint32_t FIFOCTRL_BIT_RESERVED0   :2;
                __IO  uint32_t TPSA                     :6;
                __IO  uint32_t FPM                      :2;

            }FIFOCTRL_Item;
    };
    
    union
    {
                    __IO  uint32_t SYCK;                  /*!< Offset: 0x024 SYCK Register (R/W)  */
        struct
                {
                    __IO  uint32_t SDR                  :7;
                          uint32_t SYCK_BIT_RESERVED0   :9;
                    __IO  uint32_t MCNTR                :1;
                    __IO  uint32_t MCNT                 :2;
                    __IO  uint32_t WRCKS                :1;
                    __IO  uint32_t STRR                 :1;

                }SYCK_Item;
    };
    
    union
    {
        __IO    uint32_t  RFGIO;              /*!< Offset: 0x028 RF GIO Register (R/W)  */
        struct
                {
                    __IO  uint32_t IRQ1SM               :4;
                    __IO  uint32_t IRQ1IM               :1;
                    __IO  uint32_t IRQ1OEM              :1;
                          uint32_t RFGIO_BIT_RESERVED0  :2;
                    __IO  uint32_t IRQ2SM               :4;
                    __IO  uint32_t IRQ2IM               :1;
                    __IO  uint32_t IRQ2OEM              :1;
                          uint32_t RFGIO_BIT_RESERVED1  :2;
                    __IO  uint32_t CKOS                 :4; 
                    __IO  uint32_t CKOI                 :1;
                          uint32_t RFGIO_BIT_RESERVED2  :3;
                    __IO  uint32_t CKOIOS               :1;
                    __IO  uint32_t GIO2IOS              :1;
                    __IO  uint32_t GIO1IOS              :1;
                        
                                
                }RFGIO_Item;
    };    
    
            uint32_t  RF_RESERVED1[21];   /*!< Offset: 0x02C - 0x7C Reserved */
    
    union
    {
				__IO   uint32_t  CALIBRATION;
        __O    uint32_t  CALIBRATION_WRITE;             /*!< Offset: 0x080 Calibration Register (W) */
        __I    uint32_t  CALIBRATION_READ;              /*!< Offset: 0x080 Calibration Register (R) */
        struct
                {
                    __I  uint32_t RSSC                      :1;
                    __I  uint32_t FBC                       :1;
                    __I  uint32_t VBC                       :1;
                    __I  uint32_t VCC                       :1;
                         uint32_t CALIBRATION_BIT_RESERVED0 :1;
                    __I  uint32_t SWT                       :1;
                    __I  uint32_t VBS                       :1;
                         uint32_t CALIBRATION_BIT_RESERVED1 :25;
                                
                }CALIBRATION_READ_Item;
    };        
      
    __IO    uint32_t  IFCTRL;             /*!< Offset: 0x084 IF Control Register (R/W) */
    __IO    uint32_t  VCOC;               /*!< Offset: 0x088 VCO Current Register (R/W) */
    __IO    uint32_t  VCOBAND;            /*!< Offset: 0x08C VCO BAND Register (R/W) */
            uint32_t  RF_RESERVED2[12];   /*!< Offset: 0x090 - 0x0BC Reserved */            
    __IO    uint32_t  PLL1;               /*!< Offset: 0x0C0 PLL I Register (R/W) */
    __IO    uint32_t  PLL2;               /*!< Offset: 0x0C4 PLL II Register (R/W) */
    __IO    uint32_t  PLL3;               /*!< Offset: 0x0C8 PLL III Register (R/W) */
    __IO    uint32_t  PLL4;               /*!< Offset: 0x0CC PLL IV Register (R/W) */
    __IO    uint32_t  PLL5;               /*!< Offset: 0x0D0 PLL V Register (R/W) */        
    __IO    uint32_t  PLL6;               /*!< Offset: 0x0D4 PLL VI Register (R/W) */
    __IO    uint32_t  PLL7;               /*!< Offset: 0x0D8 PLL VII Register (R/W) */
            uint32_t  RF_RESERVED3[9];    /*!< Offset: 0x0DC - 0x0FC Reserved */

    __O     uint32_t  TXCTL;             /*!< Offset: 0x100 TX Control Register (W) */
    __O     uint32_t  TXPOWER;            /*!< Offset: 0x104 TX Power Register (W) */
    __O     uint32_t  TXMOD;              /*!< Offset: 0x108 TX Modulation Register (W) */
    
            uint32_t  RF_RESERVED4[13];   /*!< Offset: 0x10C - 0x140 Reserved */

    __IO    uint32_t  RXCTL;              /*!< Offset: 0x140 RX Control Register (R/W) */
    __IO    uint32_t  RXGAIN;             /*!< Offset: 0x144 RX Gain Register (R/W) */
    
    union
    {
        __IO    uint32_t  RXDEM1;             /*!< Offset: 0x148 RX DEM I Register (R/W)*/
        struct
        {
            __IO  uint32_t IFBW         :2;
            __IO  uint32_t DMG          :2;
            __IO  uint32_t DMOS         :1;
            __IO  uint32_t DMT          :1;
            __I   uint32_t PRE_S        :1;
            __IO  uint32_t DC_SEL       :1;
            __IO  uint32_t RNUM         :4;
            __IO  uint32_t RNUM1        :3;
                  uint32_t RXDEM1_BIT_RESERVED0 :1;
            __IO  uint32_t SYNCS        :1;
            __IO  uint32_t CDRS         :2;
            __IO  uint32_t MRCKS        :1;
            __IO  uint32_t CST          :1;
            __IO  uint32_t IFSEL        :1;
                  uint32_t RXDEM1_BIT_RESERVED1 :10;
                    
        }RXDEM1_Item;
    };     
    

    __IO    uint32_t  RXDEM2;             /*!< Offset: 0x14C RX DEM II Register (R/W)*/    
		union
    {
        __IO    uint32_t  RXDEM3;              /*!< Offset: 0x150 RX DEM III Register (R/W)*/
        struct
        {
            __IO  uint32_t EARKS        :1;
            __IO  uint32_t ARC          :4;
            __IO  uint32_t ARTMS        :1;
            __IO  uint32_t VKP          :1;
            __IO  uint32_t VKM 		      :1;
            __I	  uint32_t RCR       		:4;
            __I  	uint32_t VPOAKF       :1;
            __I	  uint32_t ARTEF        :1;
                  uint32_t RXDEM3_BIT_RESERVED0 :2;
            __IO  uint32_t ARD        	:8;
            __IO  uint32_t RND         	:8;                         
        }RXDEM3_Item;
    };  
    __IO    uint32_t  SYN;                /*!< Offset: 0x154 SYN Register (R/W)*/
            uint32_t  RF_RESERVED5[2];    /*!< Offset: 0x158 - 0x160 Reserved */
    __O     uint32_t  DSSS1;              /*!< Offset: 0x160 DSSS I Register (W)*/
    __O     uint32_t  DSSS2;              /*!< Offset: 0x164 DSSS II Register (W)*/
            uint32_t  RF_RESERVED6[2];    /*!< Offset: 0x168 - 0x16C Reserved */
    __O     uint32_t  CDET;               /*!< Offset: 0x170 CDET Register (W)*/
    __O     uint32_t  CODE1;          	  /*!< Offset: 0x174 CDET Register (W)*/
    __O     uint32_t  CODE2;          	  /*!< Offset: 0x178 CDET Register (W)*/   
            uint32_t  RF_RESERVED7[17];   /*!< Offset: 0x17C - 0x1BC Reserved */
    __IO    uint32_t  DELAY;              /*!< Offset: 0x1C0 DELAY Register (R/W) */
    __IO    uint32_t  CLOCKCTRL;          /*!< Offset: 0x1C4 RF Clock Control Register (R/W) */
    __IO    uint32_t  CLOCKCTRL2;         /*!< Offset: 0x1C8 RF Clock Control II Register (W) */
    __O    	uint32_t  RFT;                /*!< Offset: 0x1CC RFT Register (W) */
       
            uint32_t  RF_RESERVED8[12];   /*!< Offset: 0x1D0 - 0x1FC Reserved */
		union
    {
        __IO    uint32_t  WORT;             /*!< Offset: 0x200 WOR & WOT Register (R/W) */
        struct
        {
            __IO  uint32_t TWOR         :1;
            __IO  uint32_t TSEL         :1;
            __IO  uint32_t WOR_EN       :1;
            __IO  uint32_t WMODE        :1;
            __IO  uint32_t WN		        :4;
                  uint32_t WORT_BIT_RESERVED0 :24;     
                    
        }WORT_Item;
    };   
    __O     uint32_t  WORT2;              /*!< Offset: 0x204 WOR & WOT Register (R/W) */
            uint32_t  RF_RESERVED9[14];   /*!< Offset: 0x208 - 0x23C Reserved */
    __IO    uint32_t  ADC;              /*!< Offset: 0x240 9-bit ADC Register control(R/W) */		 
		union
    {
				__IO   uint32_t  ADC_THRESHOLD;
        __O    uint32_t  ADC_THRESHOLD_WRITE;             /*!< Offset: 0x244 9-bit ADC Threshold Register (W) */
        __I    uint32_t  ADC_THRESHOLD_READ;              /*!< Offset: 0x244 9-bit ADC Threshold Register (R) */
        struct
                {                    
                    __I  uint32_t ADCO                       			 :9;
                         uint32_t ADC_THRESHOLD_READ_BIT_RESERVED0 :7;                    
                    __I  uint32_t ADC                       			 :8;
                         uint32_t ADC_THRESHOLD_READ_BIT_RESERVED1 :8;
                                
                }ADC_THRESHOLD_READ_Item;
    }; 
            uint32_t  RF_RESERVED10[50];  /*!< Offset: 0x248 - 0x30C Reserved */
    __IO    uint32_t  TXPKT;              /*!< Offset: 0x310 TXPKT Register (R/W) */
    __I     uint32_t  RXPKT;              /*!< Offset: 0x314 RXPKT Register (R) */
            uint32_t  RF_RESERVED11[58];  /*!< Offset: 0x324 - 0x3FC Reserved */
    __O     uint8_t   TX_DATA[RF_FIFO_SIZE];    /*!< Offset: 0x400 - 0x4FF TX DATA Register (W) */
    __I     uint8_t   RX_DATA[RF_FIFO_SIZE];    /*!< Offset: 0x500 - 0x5FF RX DATA Register (R) */
} RADIO_Type;

#define PLL1_FP_Pos      						0
#define PLL1_FP_Msk       					(0xFFFFul<<PLL1_FP_Pos)
#define PLL1_IP_Pos      						16
#define PLL1_IP_Msk       					(0x1FFul<<PLL1_iP_Pos)

#define WORT2_WRDLY_Pos      				0
#define WORT2_WRDLY_Msk       			(0x1FFul<<PLL1_FP_Pos)
#define WORT2_WORDLY_Pos      			16
#define WORT2_WORDLY_Msk       			(0x3FFul<<PLL1_FP_Pos)

#define ADC_ADCM_Pos      					0
#define ADC_ADCM_Msk       					(0x0001ul<<ADC_ADCM_Pos)
#define ADC_CDM_Pos      						1
#define ADC_CDM_Msk       					(0x0001ul<<ADC_CDM_Pos)
#define ADC_XADS_Pos      					2
#define ADC_XADS_Msk       					(0x0001ul<<ADC_XADS_Pos)
#define ADC_RADC_Pos      					3
#define ADC_RADC_Msk       					(0x0001ul<<ADC_RADC_Pos)
#define ADC_ARSSI_Pos      					4
#define ADC_ARSSI_Msk       				(0x0001ul<<ADC_ARSSI_Pos)
#define ADC_MVSEL_Pos      					8
#define ADC_MVSEL_Msk       				(0x0003ul<<ADC_MVSEL_Pos)
#define ADC_AVSEL_Pos      					10
#define ADC_AVSEL_Msk       				(0x0003ul<<ADC_AVSEL_Pos)

/* RF Strobe command */
#define CMD_SLEEP       0x80
#define CMD_IDLE        0x90
#define CMD_STBY        0xA0
#define CMD_PLL         0xB0
#define CMD_RX          0xC0
#define CMD_TX          0xD0

/* RF Reset command */
#define RF_RST          0x80
#define TXPOINT_RST     0x40
#define RXPOINT_RST     0x20
#define ADC12B_RST      0x10
#define BFC_RST         0x04

/*@}*/ /* end of group RADIO */

/*-------------------- Sleep Timer -------------------*/
/** @addtogroup SLPTIMER
    @{
*/

typedef struct
{
     __IO    uint32_t  INTERVAL;        // Offset: 0x000    SLPTIMER INTERVAL Register  (R/W) */

     __IO   uint32_t  CONTROL;          // Offset: 0x004    SLPTIMER CONTROL Register  (R/W) */
                                        //  [0]     - CE            Start SLPTIMER counting.
                                        //                              0=stop count.
                                        //                              1=start counting.
                                        //  [1]     - CKS           Clock source select
                                        //  [5]     - IF            Interrupt Flag. Write 1 to clean.
                                        //                              read 0=interrupt not occure
                                        //                              read 1=interrupt occured
                                        //                              write 0=no effect
                                        //                              write 1=clean IF
                                        //  [6]     - IE            Interrupt Enable.
                                        //                              0=disable interrupt
                                        //                              1=enable interrupt
                                        //  [7]     - ON            Turn on SLPTIMER
                                        //                              0=turn off SLPTIMER
                                        //                              1=turn on SLPTIMER
                                        //  [31]    - RST           SLPTIMER Reset  
} SLPTIMER_Type;
#define SLPTIMER_INTERVAL_Pos       0
#define SLPTIMER_INTERVAL_Msk       (0xFFFFul<<SLPTIMER_INTERVAL_Pos)

#define SLPTIMER_CONTROL_CE_Pos     0
#define SLPTIMER_CONTROL_CE_Msk     (0x1ul<<SLPTIMER_CONTROL_CE_Pos)

#define SLPTIMER_CONTROL_CKS_Pos    1
#define SLPTIMER_CONTROL_CKS_Msk    (0x3ul<<SLPTIMER_CONTROL_CKS_Pos)

#define SLPTIMER_CONTROL_IF_Pos     5
#define SLPTIMER_CONTROL_IF_Msk     (0x1ul<<SLPTIMER_CONTROL_IF_Pos)

#define SLPTIMER_CONTROL_IE_Pos     6
#define SLPTIMER_CONTROL_IE_Msk     (0x1ul<<SLPTIMER_CONTROL_IE_Pos)

#define SLPTIMER_CONTROL_ON_Pos     7
#define SLPTIMER_CONTROL_ON_Msk     (0x1ul<<SLPTIMER_CONTROL_ON_Pos)

#define SLPTIMER_CONTROL_RST_Pos    31
#define SLPTIMER_CONTROL_RST_Msk    (0x1ul<<SLPTIMER_CONTROL_RST_Pos)


/*@}*/ /* end of group SLPTIMER */

/*-------------------- AES & CCM -------------------*/

/** @addtogroup AESCCM
    @{
*/

typedef struct
{
    union
    {
        __IO   uint32_t  CONTROL;         /*!< Offset: 0x000 AES CCM CONTROL Register  (R/W) */
        __IO   uint8_t   CONTROLB[4];
    };
    union
    {
        __IO   uint32_t  NONCE[5];        /*!< Offset: 0x004 NONCE Register  (R/W) */
        __IO   uint8_t   NONCEB[20];
    };
    
    union
    {
        __IO   uint32_t  AESKEY[4];       /*!< Offset: 0x018 AES KEY Register  (R/W) */
        __IO   uint8_t   AESKEYB[16];
    };
    union
    {
        __IO   uint32_t  AESDATA[4];      /*!< Offset: 0x028 AES DATA Register  (R/W) */
        __IO   uint8_t   AESDATAB[16];
    };
    

            uint32_t  RESERVED0[50];      /*!< Offset: 0x038 - 0x0FC Reserved */
    __IO    uint8_t   RXCCM_DATA[256];    /*!< Offset: 0x100 - 0x1FF RX CCM FIFO Register  (R/W) */
    __IO    uint8_t   RXCCM_DATA2[256];   /*!< Offset: 0x200 - 0x2FF RX CCM FIFO II Register  (R/W) */
    __IO    uint8_t   TXCCM_DATA[256];    /*!< Offset: 0x300 - 0x3FF TX CCM FIFO Register  (R/W) */
    __IO    uint8_t   TXCCM_DATA2[256];   /*!< Offset: 0x400 - 0x4FF TX CCM FIFO II Register  (R/W) */
} AESCCM_Type;

/*@}*/ /* end of group AESCCM */

/*-------------------- POWER CONTROL -------------------*/

/** @addtogroup POWER
    @{
*/

typedef struct
{

    union
    {
        __IO    uint32_t  BATTERY;        /*!< Offset: 0x000 Battery Detect Register  (R/W) */
        struct
        {
            __IO    uint32_t  BDS    :   1;
            __IO    uint32_t  BVT    :   3;
            __I     uint32_t  BDF    :   1;
            __I     uint32_t  VBD    :   1; 	
        } BatteryItem;
    };
    __IO    uint32_t  FALSHCTRL;          /*!< Offset: 0x004 Flash control (R/W) */
    __IO    uint32_t  POWERCTRL1;         /*!< Offset: 0x008 Power Control I (R/W) */
    __IO    uint32_t  POWERCTRL2;         /*!< Offset: 0x00C Power Control II (R/W) */
    __IO    uint32_t  POWERCTRL3;         /*!< Offset: 0x010 Power Control III (R/W) */
    __IO    uint32_t  DCDCCTRL;           /*!< Offset: 0x014 (R/W) */
    __IO    uint32_t  DCDCCTRL2;          /*!< Offset: 0x018 (R/W) */
    __IO    uint32_t  DCDCCTRL3;          /*!< Offset: 0x01C (R/W) */
    union
    {
				__IO    uint32_t  PowerManagement;        /*!< Offset: 0x020 (R/W) */
        struct
        {
						__IO    uint32_t  PMM      :   1;
            __IO    uint32_t  STOP     :   1;
            __IO    uint32_t  SWB      :   1;
										uint32_t  PowerMangement_RESERVED1	:  1 ;
						__IO    uint32_t  AHB_Clk_Divider      :   4;
            __IO    uint32_t  CLKSEL   :   3;
						__IO    uint32_t  PM3Flag   :   1;
						__IO    uint32_t  REGAE   :   1;
						__IO    uint32_t  QDS   :   1;
						__IO    uint32_t  EQTS0   :   1;
						__IO    uint32_t  EQTS1   :   1;
										uint32_t  PowerMangement_RESERVED2	:  3 ;	
						__IO    uint32_t  PMU_SEL   :   1;
						__IO    uint32_t  BBIOS   :   1;
						__IO    uint32_t  RTCIOS   :   1;
						__IO    uint32_t  I2CIOS   :   1;
						__IO    uint32_t  IOEN   :   1;
            __IO    uint32_t  PODF     :   1;
            __IO    uint32_t  RESETF   :   1;
            __IO    uint32_t  BODF     :   1;
        } PowerManagementItem;
	};			
    __IO    uint32_t  DCDCCTRL4;    			/*!< Offset: 0x024 (R/W) */
		__IO    uint32_t  IO_CTRL0;    				/*!< Offset: 0x028 (R/W) */
		__IO    uint32_t  IO_CTRL1;    				/*!< Offset: 0x02C (R/W) */
		__IO    uint32_t  IO_CTRL2;    				/*!< Offset: 0x030 (R/W) */
		__IO    uint32_t  IO_CTRL3;    				/*!< Offset: 0x034 (R/W) */
		__IO    uint32_t  IO_CTRL4;    				/*!< Offset: 0x038 (R/W) */
		__IO    uint32_t  XRC_CTRL;				    /*!< Offset: 0x03C (R/W) */		
		__IO    uint32_t  RCCTRL1;            /*!< Offset: 0x040 RC Control I Register  (R/W) */
    __IO    uint32_t  RCCTRL2;            /*!< Offset: 0x044 RC Control II Register  (R/W) */
    __IO    uint32_t  RCTGM;              /*!< Offset: 0x048 RC Target Register  (R/W) */
		__IO    uint32_t  HDVR;               /*!< Offset: 0x04C (R/W) */
    __IO    uint32_t  IO_Ctrl5;           /*!< Offset: 0x050 (R/W) */
    __IO    uint32_t  RCFIN1;             /*!< Offset: 0x054 (R/W) */
    __IO    uint32_t  RCFIN2;             /*!< Offset: 0x058 (R/W) */
    __O     uint32_t  IO_CTRL5;          	/*!< Offset: 0x05C (W) */
    __IO    uint32_t  ECCCTRL1;	          /*!< Offset: 0x060 (R/W) */
    __IO    uint32_t  ECCCTRL2;	          /*!< Offset: 0x064 (R/W) */
    __IO    uint32_t  ECCCTRL3;	          /*!< Offset: 0x068 (R/W) */
    __IO    uint32_t  ECCCTRL4;	          /*!< Offset: 0x06C (R/W) */
    __O     uint32_t  TESTMODE;           /*!< Offset: 0x070 (W) */
    __IO    uint32_t  TDOMAIN1;           /*!< Offset: 0x074 (R/W) */
    __IO    uint32_t  TDOMAIN2;           /*!< Offset: 0x078 (R/W) */
    __O     uint32_t  DCDCCTRL6;          /*!< Offset: 0x07C (W) */
    __IO    uint32_t  WDTCTRL;            /*!< Offset: 0x080 (R/W) */
    __O     uint32_t  PWR_TA;             /*!< Offset: 0x084 (W) */
} POWER_Type;

#define POWER_BATTERY_BDS_Pos       0
#define POWER_BATTERY_BDS_Msk       (0x1ul<<POWER_BATTERY_BDS_Pos)
#define POWER_BATTERY_BVT_Pos       1
#define POWER_BATTERY_BVT_Msk       (0x7ul<<POWER_BATTERY_BVT_Pos)
#define POWER_BATTERY_BDF_Pos       4
#define POWER_BATTERY_BDF_Msk       (0x1ul<<POWER_BATTERY_BDF_Pos)

#define POWER_RCCTRL_ENCAL_Pos      0
#define POWER_RCCTRL_ENCAL_Msk      (0x1ul<<POWER_RCCTRL_ENCAL_Pos)
#define POWER_RCCTRL_TMRE_Pos       1
#define POWER_RCCTRL_TMRE_Msk       (0x1ul<<POWER_RCCTRL_TMRE_Pos)
#define POWER_RCCTRL_RCTS_Pos       2
#define POWER_RCCTRL_RCTS_Msk       (0x1ul<<POWER_RCCTRL_RCTS_Pos)
#define POWER_RCCTRL_RTCIOSEL_Pos   3
#define POWER_RCCTRL_RTCIOSEL_Msk   (0x1ul<<POWER_RCCTRL_RTCIOSEL_Pos)
#define POWER_RCCTRL_MAN_Pos        4
#define POWER_RCCTRL_MAN_Msk        (0x1ul<<POWER_RCCTRL_MAN_Pos)
#define POWER_RCCTRL_MCALS_Pos      5
#define POWER_RCCTRL_MCALS_Msk      (0x1ul<<POWER_RCCTRL_MCALS_Pos)

/*@}*/ /* end of group POWER */

/*-------------------- 12-BIT ADC -------------------*/

/** @addtogroup ADC12B
    @{
*/
typedef struct
{
    __IO    uint32_t  Control;            /*!< Offset: 0x000 12-BIT ADC Control Register (R/W) */
    union
    {
        __I    uint32_t  Value;           /*!< Offset: 0x004 Value(R) */
        struct
        {
            __I  uint16_t  ADC;
            __I  uint16_t  MVADC;
        } ValueItem;
    };
    __IO    uint32_t  ADCIOS;              /*!< Offset: 0x008 ADC IO SELECT Register (R/W) */
} ADC12B_Type;

#define ADC12B_CONTROL_ADCE_Pos     0
#define ADC12B_CONTROL_ADCE_Msk     (0x1ul<<ADC12B_CONTROL_ADCE_Pos)

#define ADC12B_CONTROL_MVS_Pos      1
#define ADC12B_CONTROL_MVS_Msk      (0x7ul<<ADC12B_CONTROL_MVS_Pos)

#define ADC12B_CONTROL_MODE_Pos     4
#define ADC12B_CONTROL_MODE_Msk     (0x1ul<<ADC12B_CONTROL_MODE_Pos)

#define ADC12B_CONTROL_CKS_Pos      5
#define ADC12B_CONTROL_CKS_Msk      (0x3ul<<ADC12B_CONTROL_CKS_Pos)

#define ADC12B_CONTROL_BUPS_Pos     7
#define ADC12B_CONTROL_BUPS_Msk     (0x1ul<<ADC12B_CONTROL_BUPS_Pos)

#define ADC12B_CONTROL_DTMP_Pos     8
#define ADC12B_CONTROL_DTMP_Msk     (0x1ul<<ADC12B_CONTROL_DTMP_Pos)

#define ADC12B_CONTROL_ENADC_Pos    9
#define ADC12B_CONTROL_ENADC_Msk    (0x1ul<<ADC12B_CONTROL_ENADC_Pos)

#define ADC12B_CONTROL_ADCYC_Pos    10
#define ADC12B_CONTROL_ADCYC_Msk    (0x1ul<<ADC12B_CONTROL_ADCYC_Pos)

#define ADC12B_CONTROL_ADIVL_Pos    11
#define ADC12B_CONTROL_ADIVL_Msk    (0x1ul<<ADC12B_CONTROL_ADIVL_Pos)

#define ADC12B_CONTROL_VADS_Pos     12
#define ADC12B_CONTROL_VADS_Msk     (0x3ul<<ADC12B_CONTROL_VADS_Pos)

#define ADC12B_CONTROL_ADCIE_Pos    15
#define ADC12B_CONTROL_ADCIE_Msk    (0x1ul<<ADC12B_CONTROL_ADCIE_Pos)

#define ADC12B_CONTROL_ADCIF_Pos    16
#define ADC12B_CONTROL_ADCIF_Msk    (0x1ul<<ADC12B_CONTROL_ADCIF_Pos)

#define ADC12B_CONTROL_ADC12RN_Pos  31
#define ADC12B_CONTROL_ADC12RN_Msk  (0x1ul<<ADC12B_CONTROL_ADC12RN_Pos)

#define ADC12B_VALUE_ADC_Pos        0
#define ADC12B_VALUE_ADC_Msk        (0xFFFul<<ADC12B_VALUE_ADC_Pos)

#define ADC12B_VALUE_MVADC_Pos      16
#define ADC12B_VALUE_MVADC_Msk      (0xFFF<<ADC12B_VALUE_MVADC_Pos)

#define ADC12B_IOSEL_EN_Pos         0
#define ADC12B_IOSEL_EN_Msk         (0x1ul<<ADC12B_IOSEL_EN_Pos)

#define ADC12B_IOSEL_CH_Pos         1
#define ADC12B_IOSEL_CH_Msk         (0x7ul<<ADC12B_IOSEL_CH_Pos)
/*@}*/ /* end of group ADC12B */

/*-------------------- SPI -------------------*/
/** @addtogroup SPI
    @{
*/
typedef struct
{
    __IO    uint32_t    CTRL;           // Offset: 0x000
    __IO    uint32_t    STATUS;         // Offset: 0x004
    __IO    uint32_t    INT;            // Offset: 0x008
    __IO    uint32_t    SSCR;           // Offset: 0x00C
    __I     uint32_t    FIFOTXSTUS;     // Offset: 0x010
    __I     uint32_t    FIFORXSTUS;     // Offset: 0x014
    __O     uint32_t    FIFOTXBUF;      // Offset: 0x018
    __I     uint32_t    FIFORXBUF;      // Offset: 0x01C
} SPI_Type;

#define SPI_CTRL_SPR0_Pos           0
#define SPI_CTRL_SPR0_Msk           (0x1ul<<SPI_CTRL_SPR0_Pos)
#define SPI_CTRL_SPR1_Pos           1
#define SPI_CTRL_SPR1_Msk           (0x1ul<<SPI_CTRL_SPR1_Pos)
#define SPI_CTRL_TYPE_Pos           2
#define SPI_CTRL_TYPE_Msk           (0x3ul<<SPI_CTRL_TYPE_Pos)
#define SPI_CTRL_CPHA_Pos           2
#define SPI_CTRL_CPHA_Msk           (0x1ul<<SPI_CTRL_CPHA_Pos)
#define SPI_CTRL_CPOL_Pos           3
#define SPI_CTRL_CPOL_Msk           (0x1ul<<SPI_CTRL_CPOL_Pos)
#define SPI_CTRL_MSTR_Pos           4
#define SPI_CTRL_MSTR_Msk           (0x1ul<<SPI_CTRL_MSTR_Pos)
#define SPI_CTRL_SPR2_Pos           5
#define SPI_CTRL_SPR2_Msk           (0x1ul<<SPI_CTRL_SPR2_Pos)
#define SPI_CTRL_SPE_Pos            6
#define SPI_CTRL_SPE_Msk            (0x1ul<<SPI_CTRL_SPE_Pos)
#define SPI_CTRL_SPIE_Pos           7
#define SPI_CTRL_SPIE_Msk           (0x1ul<<SPI_CTRL_SPIE_Pos)
#define SPI_CTRL_BITS_Pos           8
#define SPI_CTRL_BITS_Msk           (0xFul<<SPI_CTRL_BITS_Pos)
#define SPI_CTRL_FIFOEN_Pos         12
#define SPI_CTRL_FIFOEN_Msk         (0x1ul<<SPI_CTRL_FIFOEN_Pos)
#define SPI_CTRL_SFIFORST_Pos       15
#define SPI_CTRL_SFIFORST_Msk       (0x1ul<<SPI_CTRL_SFIFORST_Pos)
#define SPI_CTRL_TXFIFOIL_Pos       16
#define SPI_CTRL_TXFIFOIL_Msk       (0x7ul<<SPI_CTRL_TXFIFOIL_Pos)
#define SPI_CTRL_RXFIFOIL_Pos       20
#define SPI_CTRL_RXFIFOIL_Msk       (0x7ul<<SPI_CTRL_RXFIFOIL_Pos)

#define SPI_STATUS_SSCEN_Pos        0
#define SPI_STATUS_SSCEN_Msk        (0x1ul<<SPI_STATUS_SSCEN_Pos)
#define SPI_STATUS_MODF_Pos         4
#define SPI_STATUS_MODF_Msk         (0x1ul<<SPI_STATUS_MODF_Pos)
#define SPI_STATUS_SS_Pos           5
#define SPI_STATUS_SS_Msk           (0x1ul<<SPI_STATUS_SS_Pos)
#define SPI_STATUS_WCOL_Pos         6
#define SPI_STATUS_WCOL_Msk         (0x1ul<<SPI_STATUS_WCOL_Pos)
#define SPI_STATUS_SPIF_Pos         7
#define SPI_STATUS_SPIF_Msk         (0x1ul<<SPI_STATUS_SPIF_Pos)

#define SPI_INT_TXFIFOINTEN_Pos     0
#define SPI_INT_TXFIFOINTEN_Msk     (0x1ul<<SPI_INT_TXFIFOINTEN_Pos)
#define SPI_INT_RXFIFOINTEN_Pos     1
#define SPI_INT_RXFIFOINTEN_Msk     (0x1ul<<SPI_INT_RXFIFOINTEN_Pos)
#define SPI_INT_TXFIFOINTFLAG_Pos   4
#define SPI_INT_TXFIFOINTFLAG_Msk   (0x1ul<<SPI_INT_TXFIFOINTFLAG_Pos)
#define SPI_INT_RXFIFOINTFLAG_Pos   5
#define SPI_INT_RXFIFOINTFLAG_Msk   (0x1ul<<SPI_INT_RXFIFOINTFLAG_Pos)

#define SPI_SSCR_Pos                0
#define SPI_SSCR_Msk                (0xFFul<<SPI_SSCR_Pos)

#define SPI_FIFOTXSTUS_REM_Pos      0
#define SPI_FIFOTXSTUS_REM_Msk      (0x7ul<<SPI_FIFOTXSTUS_REM_Pos)
#define SPI_FIFOTXSTUS_EMPTY_Pos    3
#define SPI_FIFOTXSTUS_EMPTY_Msk    (0x1ul<<SPI_FIFOTXSTUS_EMPTY_Pos)
#define SPI_FIFOTXSTUS_FULL_Pos     4
#define SPI_FIFOTXSTUS_FULL_Msk     (0x1ul<<SPI_FIFOTXSTUS_FULL_Pos)
#define SPI_FIFORXSTUS_REM_Pos      0
#define SPI_FIFORXSTUS_REM_Msk      (0x7ul<<SPI_FIFORXSTUS_REM_Pos)
#define SPI_FIFORXSTUS_EMPTY_Pos    3
#define SPI_FIFORXSTUS_EMPTY_Msk    (0x1ul<<SPI_FIFORXSTUS_EMPTY_Pos)
#define SPI_FIFORXSTUS_FULL_Pos     4
#define SPI_FIFORXSTUS_FULL_Msk     (0x1ul<<SPI_FIFORXSTUS_FULL_Pos)

#define SPI_FIFOTXBUF_Pos           0
#define SPI_FIFOTXBUF_Msk           (0xFFFFul<<SPI_FIFOTXBUF_Pos)

#define SPI_FIFORXBUF_Pos           0
#define SPI_FIFORXBUF_Msk           (0xFFFFul<<SPI_FIFORXBUF_Pos)

/*@}*/ /* end of group SPI */

/*-------------------- RTC -------------------*/
/** @addtogroup RTC
    @{
*/
typedef struct
{
    __IO    uint32_t    COUNT;          // Offset: 0x000
    __IO    uint32_t    ALARM;          // Offset: 0x004
    __IO    uint32_t    CTRL;           // Offset: 0x008
    __IO    uint32_t    FLAG;           // Offset: 0x00C
    __IO    uint32_t    MANUALCOUNT;    // Offset: 0x010
		__IO    uint32_t    RTCEX;					// Offset: 0x014
} RTC_Type;

#define RTC_COUNT_TMS_Pos           0
#define RTC_COUNT_TMS_Msk           (0x7Ful<<RTC_COUNT_TMS_Pos)

#define RTC_COUNT_SEC_Pos           8
#define RTC_COUNT_SEC_Msk           (0x3Ful<<RTC_COUNT_SEC_Pos)

#define RTC_COUNT_MIN_Pos           16
#define RTC_COUNT_MIN_Msk           (0x3Ful<<RTC_COUNT_MIN_Pos)

#define RTC_COUNT_HOUR_Pos          24
#define RTC_COUNT_HOUR_Msk          (0x1Ful<<RTC_COUNT_HOUR_Pos)

#define RTC_COUNT_WEEK_Pos          29
#define RTC_COUNT_WEEK_Msk          (0x7ul<<RTC_COUNT_WEEK_Pos)

#define RTC_ALARM_MIN_Pos           0
#define RTC_ALARM_MIN_Msk           (0x3Ful<<RTC_ALARM_MIN_Pos)

#define RTC_ALARM_HOUR_Pos          8
#define RTC_ALARM_HOUR_Msk          (0x1Ful<<RTC_ALARM_HOUR_Pos)

#define RTC_CTRL_EN_Pos             0
#define RTC_CTRL_EN_Msk             (0x1ul<<RTC_CTRL_EN_Pos)

#define RTC_CTRL_ALARMIE_Pos        1
#define RTC_CTRL_ALARMIE_Msk        (0x1ul<<RTC_CTRL_ALARMIE_Pos)

#define RTC_CTRL_SECIE_Pos          2
#define RTC_CTRL_SECIE_Msk          (0x1ul<<RTC_CTRL_SECIE_Pos)

#define RTC_CTRL_HSECIE_Pos         3
#define RTC_CTRL_HSECIE_Msk         (0x1ul<<RTC_CTRL_HSECIE_Pos)

#define RTC_CTRL_TMSIE_Pos          4
#define RTC_CTRL_TMSIE_Msk         (0x1ul<<RTC_CTRL_TMSIE_Pos)

#define RTC_CTRL_MANUALCOUNTIE_Pos  5
#define RTC_CTRL_MANUALCOUNTIE_Msk  (0x1ul<<RTC_CTRL_MANUALCOUNTIE_Pos)

#define RTC_CTRL_ON_Pos             6
#define RTC_CTRL_ON_Msk             (0x1ul<<RTC_CTRL_ON_Pos)

#define RTC_CTRL_CLKSEL_Pos         7
#define RTC_CTRL_CLKSEL_Msk         (0x1ul<<RTC_CTRL_CLKSEL_Pos)

#define RTC_CTRL_MANUALCOUNTEN_Pos  8
#define RTC_CTRL_MANUALCOUNTEN_Msk  (0x1ul<<RTC_CTRL_MANUALCOUNTEN_Pos)

#define RTC_FLAG_RTC_Pos            0
#define RTC_FLAG_RTC_Msk            (0x1ul<<RTC_FLAG_RTC_Pos)

#define RTC_FLAG_ALARM_Pos          1
#define RTC_FLAG_ALARM_Msk          (0x1ul<<RTC_FLAG_ALARM_Pos)

#define RTC_FLAG_SEC_Pos            2
#define RTC_FLAG_SEC_Msk            (0x1ul<<RTC_FLAG_SEC_Pos)

#define RTC_FLAG_HSEC_Pos           3
#define RTC_FLAG_HSEC_Msk           (0x1ul<<RTC_FLAG_HSEC_Pos)

#define RTC_FLAG_TMS_Pos            4
#define RTC_FLAG_TMS_Msk           (0x1ul<<RTC_FLAG_TMS_Pos)

#define RTC_FLAG_MANUALCOUNT_Pos    5
#define RTC_FLAG_MANUALCOUNT_Msk    (0x1ul<<RTC_FLAG_MANUALCOUNT_Pos)

#define RTC_MANUALCOUNT_COUNT_Pos   0
#define RTC_MANUALCOUNT_COUNT_Msk   (0xFFFFul<<RTC_MANUALCOUNT_COUNT_Pos)

#define RTC_RTCEX_RTCDT_Pos   			0
#define RTC_RTCEX_RTCDT_Msk   			(0x1Ful<<RTC_RTCEX_RTCDT_Pos)

#define RTC_RTCEX_RTCMT_Pos   			8
#define RTC_RTCEX_RTCMT_Msk   			(0xFul<<RTC_RTCEX_RTCMT_Pos)

#define RTC_RTCEX_RTCY_Pos   				16
#define RTC_RTCEX_RTCY_Msk   				(0x7Ful<<RTC_RTCEX_RTCY_Pos)

/*@}*/ /* end of group RTC */
/*-------------------- Device ID -------------------*/

/** @addtogroup Device ID 
    @{
*/
typedef struct
{
    __O    uint32_t  RF_Device_ID;            /*!< Offset: 0x000 Device ID Register (R/W) */
} IP_Information_Type;


/*@}*/ /* end of group Device ID */

/*-------------------- PN Code -------------------*/

/** @addtogroup PN Code
    @{
*/

typedef struct
{

    __IO    uint32_t  PNCFG1;          /*!< Offset: 0x000 PN Code CFGl (R/W) */
    __IO    uint32_t  PNCFG2;          /*!< Offset: 0x004 PN Code CFG2 (R/W) */
    __IO    uint32_t  PNCFG3;          /*!< Offset: 0x008 PN Code CFG3 (R/W) */
    __IO    uint32_t  PNCFG4;          /*!< Offset: 0x00C PN Code CFG4 (R/W) */
    __IO    uint32_t  PNCFG5;          /*!< Offset: 0x010 PN Code CFG5 (R/W) */
    __IO    uint32_t  PNCFG6;          /*!< Offset: 0x014 PN Code CFG6 (R/W) */
    __IO    uint32_t  PNCFG7;          /*!< Offset: 0x018 PN Code CFG7 (R/W) */
    __IO    uint32_t  PNCFG8;          /*!< Offset: 0x01C PN Code CFG8 (R/W) */
    __O     uint32_t  PNCFG9;          /*!< Offset: 0x020 PN Code CFG9 (W) */
		__O     uint32_t  Misc1;           /*!< Offset: 0x024 Misc1 (W) */
		__O     uint32_t  Misc2;           /*!< Offset: 0x028 Misc2 (W) */   
    __O     uint32_t  Misc3;           /*!< Offset: 0x02C Misc3 (W) */
    __IO    uint32_t  CSMA1;           /*!< Offset: 0x030 CSMA1 (R/W) */
    __IO    uint32_t  DCMON1;          /*!< Offset: 0x034 DCMON1 (R/W) */	
    __IO    uint32_t  DCMON2;          /*!< Offset: 0x038 DCMON2 (R/W) */
    __IO    uint32_t  MBUS1;           /*!< Offset: 0x03C MBUS1 (R/W) */
    __O    	uint32_t  MBUS2;           /*!< Offset: 0x040 MBUS2 (W) */
    __O    	uint32_t  MBUS3;           /*!< Offset: 0x044 MBUS3 (W) */		
    __O    	uint32_t  PS1;             /*!< Offset: 0x048 Preamble Sence1 (W) */
    __O    	uint32_t  PS2;             /*!< Offset: 0x04C Preamble Sence2 (W) */			
	

} PN_Code_Type;

/* PN Register Definitions */

#define PNCFG1_Delay_PN_Pos     	 0
#define PNCFG1_Delay_PN_Msk     	 (0xFFFFul<<PNCFG1_Delay_PN_Pos)
#define PNCFG1_PN_REG_Keep_Pos  	 16
#define PNCFG1_PN_REG_Keep_Msk  	 (0x1ul<<PNCFG1_PN_REG_Keep_Pos)
#define PNCFG1_Manual_Poly_1_Pos   17
#define PNCFG1_Manual_Poly_1_Msk   (0x1ul<<PNCFG1_Manual_Poly_1_Pos)
#define PNCFG1_PN_WOT_En_Pos       18
#define PNCFG1_PN_WOT_En_Msk       (0x1ul<<PNCFG1_PN_WOT_En_Pos)
#define PNCFG1_PN_WOR_En_Pos       19
#define PNCFG1_PN_WOR_En_Msk       (0x1ul<<PNCFG1_PN_WOR_En_Pos)
#define PNCFG1_MAN_Torence_Pos     20
#define PNCFG1_MAN_Torence_Msk     (0xFul<<PNCFG1_MAN_Torence_Pos)
#define PNCFG1_MS_Torence_Pos      24
#define PNCFG1_MS_Torence_Msk      (0xFul<<PNCFG1_MS_Torence_Pos)
#define PNCFG1_Manual_Poly_5_Pos   28
#define PNCFG1_Manual_Poly_5_Msk   (0x1ul<<PNCFG1_Manual_Poly_5_Pos)
#define PNCFG1_Manual_Poly_4_Pos   29
#define PNCFG1_Manual_Poly_4_Msk   (0x1ul<<PNCFG1_Manual_Poly_4_Pos)
#define PNCFG1_Manual_Poly_3_Pos   30
#define PNCFG1_Manual_Poly_3_Msk   (0x1ul<<PNCFG1_Manual_Poly_3_Pos)
#define PNCFG1_Manual_Poly_2_Pos   31
#define PNCFG1_Manual_Poly_2_Msk   (0x1ul<<PNCFG1_Manual_Poly_2_Pos)
#define PNCFG1_HAC_WOR_State0_Pos  11
#define PNCFG1_HAC_WOR_State0_Msk  (0x1ul<<PNCFG1_HAC_WOR_State0_Pos)
#define PNCFG1_HAC_WOR_State1_Pos  12
#define PNCFG1_HAC_WOR_State1_Msk  (0x1ul<<PNCFG1_HAC_WOR_State1_Pos)
#define PNCFG1_HAC_WOR_State2_Pos  13
#define PNCFG1_HAC_WOR_State2_Msk  (0x1ul<<PNCFG1_HAC_WOR_State2_Pos)
#define PNCFG1_HOP_ID_Pos  				 14
#define PNCFG1_HOP_ID_Msk          (0x1ul<<PNCFG1_HOP_ID_Pos)
#define PNCFG1_WakeupBit_Done_Pos  15
#define PNCFG1_WakeupBit_Done_Msk  (0x1ul<<PNCFG1_WakeupBit_Done_Pos)
#define PNCFG1_Flag_Poly_0_Pos     16
#define PNCFG1_Flag_Poly_0_Msk     (0x1ul<<PNCFG1_Flag_Poly_0_Pos)
#define PNCFG1_Flag_Poly_1_Pos     17
#define PNCFG1_Flag_Poly_1_Msk     (0x1ul<<PNCFG1_Flag_Poly_1_Pos)
#define PNCFG1_Flag_Poly_2_Pos     18
#define PNCFG1_Flag_Poly_2_Msk     (0x1ul<<PNCFG1_Flag_Poly_2_Pos)
#define PNCFG1_Flag_Poly_3_Pos     19
#define PNCFG1_Flag_Poly_3_Msk     (0x1ul<<PNCFG1_Flag_Poly_3_Pos)
#define PNCFG1_Flag_Poly_4_Pos     20
#define PNCFG1_Flag_Poly_4_Msk     (0x1ul<<PNCFG1_Flag_Poly_4_Pos)
#define PNCFG1_Flag_Poly_5_Pos     21
#define PNCFG1_Flag_Poly_5_Msk     (0x1ul<<PNCFG1_Flag_Poly_5_Pos)

#define PNCFG9_POLY_0_En_Pos     	 0
#define PNCFG9_POLY_0_En_Msk     	 (0x1ul<<PNCFG9_POLY_0_En_Pos)
#define PNCFG9_POLY_1_En_Pos     	 1
#define PNCFG9_POLY_1_En_Msk     	 (0x1ul<<PNCFG9_POLY_1_En_Pos)
#define PNCFG9_POLY_2_En_Pos     	 2
#define PNCFG9_POLY_2_En_Msk     	 (0x1ul<<PNCFG9_POLY_2_En_Pos)
#define PNCFG9_POLY_3_En_Pos     	 3
#define PNCFG9_POLY_3_En_Msk     	 (0x1ul<<PNCFG9_POLY_3_En_Pos)
#define PNCFG9_POLY_4_En_Pos     	 4
#define PNCFG9_POLY_4_En_Msk     	 (0x1ul<<PNCFG9_POLY_4_En_Pos)
#define PNCFG9_POLY_5_En_Pos     	 5
#define PNCFG9_POLY_5_En_Msk     	 (0x1ul<<PNCFG9_POLY_5_En_Pos)
#define PNCFG9_MAN_Long1_Pos     	 6
#define PNCFG9_MAN_Long1_Msk     	 (0x1ul<<PNCFG9_MAN_Long1_Pos)
#define PNCFG9_MAN_Long_Pos     	 7
#define PNCFG9_MAN_Long_Msk     	 (0x1ul<<PNCFG9_MAN_Long_Pos)
#define PNCFG9_FEC_WOR_En_Pos      8
#define PNCFG9_FEC_WOR_En_Msk      (0x1ul<<PNCFG9_FEC_WOR_En_Pos)
#define PNCFG9_PN_Rec_bits_Pos     9
#define PNCFG9_PN_Rec_bits_Msk     (0x3Ful<<PNCFG9_PN_Rec_bits_Pos)
#define PNCFG9_PN_n_Pos     			 16
#define PNCFG9_PN_n_Msk     			 (0x1Ful<<PNCFG9_PN_n_Pos)
#define PNCFG9_ENRX_S_Pos    			 21
#define PNCFG9_ENRX_S_Msk     		 (0x1Ful<<PNCFG9_ENRX_S_Pos)
#define PNCFG9_MAN_BSBE_Pos    		 22
#define PNCFG9_MAN_BSBE_Msk     	 (0x1ul<<PNCFG9_MAN_BSBE_Pos)
#define PNCFG9_MAN_PNS_Pos    		 23
#define PNCFG9_MAN_PNS_Msk     	   (0x1ul<<PNCFG9_MAN_PNS_Pos)
#define PNCFG9_BSBE_S_Pos    		   24
#define PNCFG9_BSBE_S_Msk     	   (0x1ul<<PNCFG9_BSBE_S_Pos)
#define PNCFG9_MAN_PN_RX_Pos 		   25
#define PNCFG9_MAN_PN_RX_Msk   	   (0x1ul<<PNCFG9_MAN_PN_RX_Pos)
#define PNCFG9_MAN_FEC_RX_Pos 		 26
#define PNCFG9_MAN_FEC_RX_Msk   	 (0x1ul<<PNCFG9_MAN_FEC_RX_Pos)
#define PNCFG9_TST_MODE_Pos 		   27
#define PNCFG9_TST_MODE_Msk   	   (0x1ul<<PNCFG9_TST_MODE_Pos)
#define PNCFG9_WakeupBit_PolyID_Pos 28
#define PNCFG9_WakeupBit_PolyID_Msk (0x7ul<<PNCFG9_WakeupBit_PolyID_Pos)
#define PNCFG9_WakeupBit_En_Pos 	31
#define PNCFG9_WakeupBit_En_Msk   (0x1ul<<PNCFG9_WakeupBit_En_Pos)

#define Misc1_WATCH_Sel_Pos 			0
#define Misc1_WATCH_Sel_Msk   		(0x3ul<<Misc1_WATCH_Sel_Pos)
#define Misc1_MAN_Rec_bits_Pos 		2
#define Misc1_MAN_Rec_bits_Msk   	(0x3Ful<<Misc1_MAN_Rec_bits_Pos)
#define Misc1_MAN_OK_S_Pos 		    8
#define Misc1_MAN_OK_S_Msk   	    (0x1ul<<Misc1_MAN_OK_S_Pos)
#define Misc1_MAN_NEW_Pos 		    9
#define Misc1_MAN_NEW_Msk   	    (0x1ul<<Misc1_MAN_NEW_Pos)
#define Misc1_MAN_BYP_Pos 		    10
#define Misc1_MAN_BYP_Msk   	    (0x1ul<<Misc1_MAN_BYP_Pos)
#define Misc1_RAW_Sel_Pos 		    11
#define Misc1_RAW_Sel_Msk   	    (0x1ul<<Misc1_RAW_Sel_Pos)
#define Misc1_MAN_START_Pos 		  12
#define Misc1_MAN_START_Msk   	  (0x1ul<<Misc1_MAN_START_Pos)
#define Misc1_MAN_Sel_Pos 		    13
#define Misc1_MAN_Sel_Msk   	    (0x1ul<<Misc1_MAN_Sel_Pos)
#define Misc1_EXDIS_Pos 		      14
#define Misc1_EXDIS_Msk   	      (0x1ul<<Misc1_EXDIS_Pos)
#define Misc1_PN_START_Pos 		    15
#define Misc1_PN_START_Msk   	    (0x1ul<<Misc1_PN_START_Pos)
#define Misc1_DC_DIFF_Pos 		    16
#define Misc1_DC_DIFF_Msk   	    (0x1Ful<<Misc1_DC_DIFF_Pos)
#define Misc1_DC_DTH_Pos 		      21
#define Misc1_DC_DTH_Msk   	      (0x1ul<<Misc1_DC_DTH_Pos)
#define Misc1_CD_Mode_Pos 		    22
#define Misc1_CD_Mode_Msk   	    (0x1ul<<Misc1_CD_Mode_Pos)
#define Misc1_FREQ_HOP_N_Pos 		  24
#define Misc1_FREQ_HOP_N_Msk   	  (0x1Ful<<Misc1_FREQ_HOP_N_Pos)

#define Misc2_HOP_Delay_Pos 		  0
#define Misc2_HOP_Delay_Msk   	  (0xFFul<<Misc2_HOP_Delay_Pos)
#define Misc2_PN0_Test_Mux_Pos 		8
#define Misc2_PN0_Test_Mux_Msk   	(0x1ul<<Misc2_PN0_Test_Mux_Pos)
#define Misc2_PN1_Test_Mux_Pos 		9
#define Misc2_PN1_Test_Mux_Msk   	(0x7ul<<Misc2_PN1_Test_Mux_Pos)
#define Misc2_PN2_Test_Mux2_Pos 	12
#define Misc2_PN2_Test_Mux2_Msk   (0x7ul<<Misc2_PN2_Test_Mux2_Pos)
#define Misc2_PN_Test_En_Pos 			15
#define Misc2_PN_Test_En_Msk   		(0x7ul<<Misc2_PN_Test_En_Pos)
#define Misc2_DC_HOLD_Pos 				16
#define Misc2_DC_HOLD_Msk   		  (0x1ul<<Misc2_DC_HOLD_Pos)
#define Misc2_DLYS_Pos 						17
#define Misc2_DLYS_Msk   		  		(0x1ul<<Misc2_DLYS_Pos)
#define Misc2_DLY_NUM_Pos					18
#define Misc2_DLY_NUM_Msk		  		(0x3Ful<<Misc2_DLY_NUM_Pos)
#define Misc2_MAN_NHOLD_Pos				24
#define Misc2_MAN_NHOLD_Msk		  	(0x1ul<<Misc2_MAN_NHOLD_Pos)
#define Misc2_DLYPAR_Pos				  25
#define Misc2_DLYPAR_Msk		  	  (0x3Ful<<Misc2_DLYPAR_Pos)

#define Misc3_FP2_Pos				  		0
#define Misc3_FP2_Msk		  	      (0xFFFFul<<Misc3_FP2_Pos)
#define Misc3_IP2_Pos				  		16
#define Misc3_IP2_Msk		  	      (0x1FFul<<Misc3_IP2_Pos)

#define CSMA1_BATL_Pos				  	0
#define CSMA1_BATL_Msk		  	    (0x1ul<<CSMA1_ERXS_Pos)
#define CSMA1_ERXS_Pos				  	1
#define CSMA1_ERXS_Msk		  	    (0x1ul<<CSMA1_ERXS_Pos)
#define CSMA1_CCA_IVL_Pos				  2
#define CSMA1_CCA_IVL_Msk		  	  (0xFul<<CSMA1_CCA_IVL_Pos)
#define CSMA1_MVRSSI_Pos				  6
#define CSMA1_MVRSSI_Msk		  	  (0x3ul<<CSMA1_MVRSSI_Pos)
#define CSMA1_TOS_Pos				      8
#define CSMA1_TOS_Msk		  	      (0x1ul<<CSMA1_TOS_Pos)
#define CSMA1_MAXNB_Pos			      9
#define CSMA1_MAXNB_Msk		  	    (0x7ul<<CSMA1_MAXNB_Pos)
#define CSMA1_CCAM_Pos			      12
#define CSMA1_CCAM_Msk		  	    (0x3ul<<CSMA1_CCAM_Pos)
#define CSMA1_CDT_Pos			      	14
#define CSMA1_CDT_Msk		  	    	(0x3ul<<CSMA1_CDT_Pos)
#define CSMA1_ENCS_Pos			      16
#define CSMA1_ENCS_Msk		  	    (0x1ul<<CSMA1_ENCS_Pos)
#define CSMA1_SLOT_Pos			      17
#define CSMA1_SLOT_Msk		  	    (0x1ul<<CSMA1_SLOT_Pos)
#define CSMA1_MAXBE_Pos			      18
#define CSMA1_MAXBE_Msk		  	    (0xFul<<CSMA1_MAXBE_Pos)
#define CSMA1_MINBE_Pos			      24
#define CSMA1_MINBE_Msk		  	    (0xFul<<CSMA1_MINBE_Pos)
#define CSMA1_EDS_Pos			      	28
#define CSMA1_EDS_Msk		  	    	(0x1ul<<CSMA1_EDS_Pos)
#define CSMA1_SET_Pos			      	29
#define CSMA1_SET_Msk		  	    	(0x1ul<<CSMA1_SET_Pos)
#define CSMA1_CCAS_Pos			      30
#define CSMA1_CCAS_Msk		  	    (0x1ul<<CSMA1_CCAS_Pos)
#define CSMA1_ADCCA_Pos			      0
#define CSMA1_ADCCA_Msk		  	    (0x1FFul<<CSMA1_ADCCA_Pos)

#define MBUS1_SPAR1_Pos			      0
#define MBUS1_SPAR1_Msk		  	    (0x7Ful<<MBUS1_SPAR1_Pos)
#define MBUS1_SRTS_Pos			      7
#define MBUS1_SRTS_Msk		  	    (0x1ul<<MBUS1_SRTS_Pos)
#define MBUS1_SRAR2_Pos			      8
#define MBUS1_SRAR2_Msk		  	    (0xFul<<MBUS1_SRTS_Pos)
#define MBUS1_TEDS_Pos			      12
#define MBUS1_TEDS_Msk		  	    (0x1ul<<MBUS1_TEDS_Pos)
#define MBUS1_LM_ON_Pos			      13
#define MBUS1_LM_ON_Msk		  	    (0x1ul<<MBUS1_LM_ON_Pos)
#define MBUS1_P2_SEL_Pos			    14
#define MBUS1_P2_SEL_Msk		  	  (0x1ul<<MBUS1_P2_SEL_Pos)
#define MBUS1_MBUS_Pos			    	15
#define MBUS1_MBUS_Msk		  	  	(0x1ul<<MBUS1_MBUS_Pos)
#define MBUS1_CPAR1_Pos			    	16
#define MBUS1_CPAR1_Msk		  	  	(0x3Ful<<MBUS1_CPAR1_Pos)
#define MBUS1_CPAR2_Pos			    	22
#define MBUS1_CPAR2_Msk		  	  	(0x3Ful<<MBUS1_CPAR2_Pos)
#define MBUS1_MSCNT_Pos			    	28
#define MBUS1_MSCNT_Msk		  	  	(0xFul<<MBUS1_MSCNT_Pos)

#define PS1_PE_Pos			    	    0
#define PS1_PE_Msk		  	  	    (0xFFFFul<<PS1_PE_Pos)
#define PS1_PS_MODE_Pos			    	16
#define PS1_PS_MODE_Msk		  	  	(0x1ul<<PS1_PS_MODE_Pos)
#define PS1_PRE_HOLD_Pos			    17
#define PS1_PRE_HOLD_Msk		  	  (0x1ul<<PS1_PRE_HOLD_Pos)
#define PS1_PROAK_SEL_Pos			    18
#define PS1_PROAK_SEL_Msk		  	  (0x1ul<<PS1_PROAK_SEL_Pos)
#define PS1_PE_ETH_Pos			      19
#define PS1_PE_ETH_Msk		  	    (0x7ul<<PS1_PE_ETH_Pos)
#define PS1_PE_MODE_Pos			      22
#define PS1_PE_MODE_Msk		  	    (0x3ul<<PS1_PE_MODE_Pos)
#define PS1_INTES_Pos			      	24
#define PS1_INTES_Msk		  	      (0x1ul<<PS1_INTES_Pos)
#define PS1_CD_S_Pos			      	25
#define PS1_CD_S_Msk		  	      (0x1ul<<PS1_CD_S_Pos)
#define PS1_SYN_Sel_Pos			      26
#define PS1_SYN_Sel_Msk	  	      (0x1ul<<PS1_SYN_Sel_Pos)
#define PS1_TOUT_Sel_Pos			    27
#define PS1_TOUT_Sel_Msk	  	    (0x1ul<<PS1_TOUT_Sel_Pos)
#define PS1_TOUT1_Sel_Pos			    28
#define PS1_TOUT1_Sel_Msk	  	    (0x1ul<<PS1_TOUT1_Sel_Pos)

#define MBUS2_MDTC1_Pos			      0
#define MBUS2_MDTC1_Msk		  	    (0xFFFul<<MBUS2_MDTC1_Pos)
#define MBUS2_MAN_DTC1_Pos			  12
#define MBUS2_MAN_DTC1_Msk		  	(0x1ul<<MBUS2_MAN_DTC1_Pos)
#define MBUS2_STH_Pos			  			13
#define MBUS2_STH_Msk		  	      (0x7ul<<MBUS2_STH_Pos)
#define MBUS2_MDTC_Pos			  		16
#define MBUS2_MDTC_Msk		  	    (0xFFFul<<MBUS2_MDTC_Pos)
#define MBUS2_MAN_DTC_Pos			  	28
#define MBUS2_MAN_DTC_Msk		  	  (0x1ul<<MBUS2_MAN_DTC_Pos)
#define MBUS2_RSPRDS_Pos			  	29
#define MBUS2_RSPRDS_Msk		  	  (0x1ul<<MBUS2_RSPRDS_Pos)
#define MBUS2_MAN_STH_Pos			  	30
#define MBUS2_MAN_STH_Msk		  	  (0x1ul<<MBUS2_MAN_STH_Pos)

#define MBUS3_SUMEPH_TH_Pos			  0
#define MBUS3_SUMEPH_TH_Msk		  	(0x3FFul<<MBUS3_SUMEPH_TH_Pos)
#define MBUS3_RX_SEL_Pos			    10
#define MBUS3_RX_SEL_Msk		  	  (0xFul<<MBUS3_RX_SEL_Pos)
#define MBUS3_DTTH_Pos			    	16
#define MBUS3_DTTH_Msk		  	    (0xFFul<<MBUS3_DTTH_Pos)
#define MBUS3_MVPNS_Pos			    	24
#define MBUS3_MVPNS_Msk		  	    (0x1ul<<MBUS3_MVPNS_Pos)
#define MBUS3_AVPNS_Pos			    	25
#define MBUS3_AVPNS_Msk		  	    (0x1ul<<MBUS3_AVPNS_Pos)
#define MBUS3_DLHS_Pos			    	26
#define MBUS3_DLHS_Msk		  	    (0x1ul<<MBUS3_DLHS_Pos)
#define MBUS3_DC_NUMS_Pos			    27
#define MBUS3_DC_NUMS_Msk		  	  (0x3ul<<MBUS3_DC_NUMS_Pos)
#define MBUS3_DCLKC_SEL_Pos			  29
#define MBUS3_DCLKC_SEL_Msk		  	(0x1ul<<MBUS3_DCLKC_SEL_Pos)
#define MBUS3_MAN_DC_Pos			    30
#define MBUS3_MAN_DC_Msk		  		(0x1ul<<MBUS3_MAN_DC_Pos)

#define PS2_TCNT1_Reg_Pos			    0
#define PS2_TCNT1_Reg_Msk		  	  (0xFFul<<PS2_TCNT1_Reg_Pos)
#define PS2_TCNT_Reg_Pre_Pos			8
#define PS2_TCNT_Reg_Pre_Msk		  (0xFFul<<PS2_TCNT_Reg_Pre_Pos)
#define PS2_TCNT0_Reg_Pos			    16
#define PS2_TCNT0_Reg_Msk		  	  (0x1FFul<<PS2_TCNT0_Reg_Pos)
#define PS2_SYNC_OFS_Pos			    25
#define PS2_SYNC_OFS_Msk		  	  (0x7Ful<<PS2_SYNC_OFS_Pos)

/*@}*/ /* end of PN Code */

/*-------------------- CHACHE -------------------*/

/** @addtogroup CACHE
    @{
*/

typedef struct
{
	  union
    {
        __IO    uint32_t  CCR;             /*!< Offset: 0x000 CCR (R/W) */
        struct
        {
            __IO    uint32_t  EN    					:   1;
            __IO    uint32_t  INV_REQ    			:   1;
            __IO    uint32_t  POW_REQ    			:   1;
            __IO    uint32_t  SET_MAN_POW     :   1; 	
						__IO    uint32_t  SET_MAN_INV     :   1; 	
						__IO    uint32_t  SET_PREFETCH    :   1; 	
						__IO    uint32_t  STATISTIC_EN    :   1; 	
        } CCRItem;
    };
		union
    {
				__I     uint32_t  SR;          		 /*!< Offset: 0x004 SR (R) */
				struct
        {
            __I    uint32_t  CS    					  :   2;
            __I    uint32_t  INV_STAT    		  :   1;
						__I    uint32_t  SR_BIT_RESERVED0 :   1;
            __I    uint32_t  POW_STAT    		  :   1;            
        } SRItem;
    };
    __IO    uint32_t  IRQMASK;         /*!< Offset: 0x008 IRQMASK (R/W) */
    __IO    uint32_t  IRQSRAR;         /*!< Offset: 0x00C IRQSTAT (R/W) */
    __I	    uint32_t  HWPARAMS;        /*!< Offset: 0x010 HWPARAMS (R) */
    __IO    uint32_t  CSHR;            /*!< Offset: 0x014 CSHR (R/W) */
    __IO    uint32_t  CSMR;            /*!< Offset: 0x018 CSMR (R/W) */	

} CACHE_Type;

/* CACHE Register Definitions */

#define SR_CS_Pos 								0
#define SR_CS_Msk   							(0x3ul<<SR_CS_Pos)
#define SR_INV_STAT_Pos 					2
#define SR_INV_STAT_Msk   				(0x1ul<<SR_INV_STAT_Pos)
#define SR_POW_STAT_Pos 		   		4
#define SR_POW_STAT_Msk   	    	(0x1ul<<SR_POW_STAT_Pos)

/*@}*/ /* end of CACHE */

/******************************************************************************/
/*                         Peripheral memory map                              */
/******************************************************************************/
/** @addtogroup PERIPHERAL_MEM_MAP Peripheral Memory Map
    @{
*/
#define TIMER0_BASE         0x40000000
#define TIMER1_BASE         0x40001000
#define DUALTIMER1_BASE     0x40002000
#define DUALTIMER2_BASE     0x40002020
#define UART0_BASE          0x40004000
#define UART1_BASE          0x40005000
#define UART2_BASE          0x40006000
#define WATCHDOG_BASE       0x40008000
#define GPIO0_BASE          0x40010000
#define GPIO1_BASE          0x40011000
#define SYSCTRL_BASE        0x4001F000
#define I2CM_BASE           0x50003000
#define I2CS_BASE           0x50003800
#define PWM0_BASE           0x50004000
#define PWM1_BASE           0x50004100
#define PWM2_BASE           0x50004200
#define PWM3_BASE           0x50004300
#define FLASH_BASE          0x4001F100
#define POWER_BASE          0x50000000
#define RADIO_BASE          0x50001000
#define SPI_BASE            0x50002000
#define RTC_BASE            0x50005000
#define SLPTIMER_BASE       0x50006000
#define SLPTIMER1_BASE      0x50006010
#define PN_Code_BASE        0x50007000
#define ADC12BITS_BASE      0x50008000
#define AESCCM_BASE         0x5000C000
#define IP_Information_BASE 0x5000F000
#define CACHE_BASE 					0x50010000
/*@}*/ /* end of group PERIPHERAL_MEM_MAP */

/******************************************************************************/
/*                         Peripheral Definitions                             */
/******************************************************************************/
/** @addtogroup PERIPHERAL Peripheral Definitions
    @{
*/
#define TIMER0          ((TIMER_Type        *)  TIMER0_BASE)
#define TIMER1          ((TIMER_Type        *)  TIMER1_BASE)
#define DUALTIMER1      ((DUALTIMER_Type    *)  DUALTIMER1_BASE)
#define DUALTIMER2      ((DUALTIMER_Type    *)  DUALTIMER2_BASE)
#define UART0           ((UART_Type         *)  UART0_BASE)
#define UART1           ((UART_Type         *)  UART1_BASE)
#define UART2           ((UART_Type         *)  UART2_BASE)
#define WATCHDOG        ((WATCHDOG_Type     *)  WATCHDOG_BASE)
#define GPIO0           ((GPIO_Type         *)  GPIO0_BASE)
#define GPIO1           ((GPIO_Type         *)  GPIO1_BASE)
#define SYSCTRL         ((SYSCON_Type       *)  SYSCTRL_BASE )
#define I2CM            ((I2CM_Type         *)  I2CM_BASE)
#define I2CS            ((I2CS_Type         *)  I2CS_BASE)
#define PWM0            ((PWM_Type          *)  PWM0_BASE)
#define PWM1            ((PWM_Type          *)  PWM1_BASE)
#define PWM2            ((PWM_Type          *)  PWM2_BASE)
#define PWM3            ((PWM_Type          *)  PWM3_BASE)
#define FLASH           ((Flash_Type        *)  FLASH_BASE)
#define POWER           ((POWER_Type        *)  POWER_BASE)
#define RADIO           ((RADIO_Type        *)  RADIO_BASE)
#define SPI             ((SPI_Type          *)  SPI_BASE)
#define RTC             ((RTC_Type          *)  RTC_BASE)
#define SLPTIMER0	      ((SLPTIMER_Type     *)  SLPTIMER_BASE)
#define SLPTIMER1	      ((SLPTIMER_Type     *)  SLPTIMER1_BASE)
#define PN_Code         ((PN_Code_Type      *)  PN_Code_BASE)
#define ADC12B          ((ADC12B_Type       *)  ADC12BITS_BASE)
#define AESCCM          ((AESCCM_Type       *)  AESCCM_BASE)
#define IP_Information  ((IP_Information_Type   *)  IP_Information_BASE)
#define CACHE					  ((CACHE_Type  			*)  CACHE_BASE)
/*@}*/ /* end of group PERIPHERAL */

#endif
